package interpol

import (
	"fmt"
	"net"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/suite"
)

const testCorrectToken = "test_correct_token"
const responseJSON = `{
	"user_id": "test_user_id",
	"login": "test_login",
    "scopes": [],
    "expires_in": 5499027
}`

type InterpolClientTest struct {
	suite.Suite
}

func TestRunInterpolClientTestSuite(t *testing.T) {
	suite.Run(t, new(InterpolClientTest))
}

func mockTestServer(f func(w http.ResponseWriter, r *http.Request)) *httptest.Server {
	server := httptest.NewUnstartedServer(http.HandlerFunc(f))
	l, _ := net.Listen("tcp", "127.0.0.1:3001")
	server.Listener = l

	return server
}

func (suite *InterpolClientTest) TestGetClientInfo_happyPath() {
	server := mockTestServer(func(rw http.ResponseWriter, req *http.Request) {
		// Test request parameters
		suite.Equal(req.URL.String(), "/oauth2/validate")
		suite.Equal(req.Method, "GET")

		// Send response to be tested
		if req.Header.Get("Authorization") == fmt.Sprintf("OAuth %s", testCorrectToken) {
			rw.WriteHeader(http.StatusOK)
			_, err := rw.Write([]byte(responseJSON))
			suite.Nil(err)
			return
		}
		http.Error(rw, "invalid token", http.StatusBadRequest)
	})
	suite.NotNil(server)
	server.Start()
	testClient := &clientImpl{
		interpolHost: server.URL,
		httpClient:   server.Client(),
	}

	defer server.Close()
	authResp, err := testClient.GetClientInfo(testCorrectToken)
	suite.NoError(err)
	suite.NotNil(authResp)
	suite.Equal("test_user_id", authResp.UserID)
	suite.Equal("test_login", authResp.Login)
}

func (suite *InterpolClientTest) TestGetClientInfo_nil_response() {
	server := mockTestServer(func(rw http.ResponseWriter, req *http.Request) {
		// Test request parameters
		suite.Equal(req.URL.String(), "/oauth2/validate")
		suite.Equal(req.Method, "GET")
	})
	suite.NotNil(server)
	server.Start()
	testClient := &clientImpl{
		interpolHost: server.URL,
		httpClient:   server.Client(),
	}

	defer server.Close()
	authResp, err := testClient.GetClientInfo(testCorrectToken)
	suite.Error(err, "failed to talk to identity")
	suite.Nil(authResp)
}

func (suite *InterpolClientTest) TestGetClientInfo_invalid_token() {
	server := mockTestServer(func(rw http.ResponseWriter, req *http.Request) {
		// Test request parameters
		suite.Equal(req.URL.String(), "/oauth2/validate")
		suite.Equal(req.Method, "GET")

		// Send response to be tested
		if req.Header.Get("Authorization") == fmt.Sprintf("OAuth %s", testCorrectToken) {
			rw.WriteHeader(http.StatusOK)
			_, err := rw.Write([]byte(responseJSON))
			suite.Nil(err)
		} else {
			http.Error(rw, "invalid token", http.StatusBadRequest)
		}
	})
	suite.NotNil(server)
	server.Start()
	testClient := &clientImpl{
		interpolHost: server.URL,
		httpClient:   server.Client(),
	}

	defer server.Close()
	authResp, err := testClient.GetClientInfo("some token")
	suite.Error(err, "token not associated with client")
	suite.Nil(authResp)
}
