package models

import (
	"encoding/json"
	"errors"
)

var DefaultServiceErr = `{"code"":"service_err","message":"service is encountering an unexpected issue"}`
var DefaultDisconnectErr = errors.New("service is encountering an unexpected issue, please try to reconnect")

type ErrAction int

// server action based on specific error type
const (
	ActionReconnect  ErrAction = 1
	ActionDisconnect ErrAction = 2
	ActionDoNothing  ErrAction = 3
)

type ErrField string
type ErrType string

const (
	SessionIDErr     ErrField = "session_id"
	EnvErr           ErrField = "env"
	TokenErr         ErrField = "token"
	BroadcasterIDErr ErrField = "broadcaster_ids"
	GameIDErr        ErrField = "game_id"
	DataErr          ErrField = "data"
	MetadataErr      ErrField = "_metadata"
	TagErr           ErrField = "_metadata.tags"
	ActiveErr        ErrField = "_metadata.active"

	ServerErr      ErrType = "server_error"
	AuthErr        ErrType = "auth_err"
	BadInput       ErrType = "bad_input"
	BadRequestRate ErrType = "bad_request_rate"
)

type ErrResp struct {
	Code    string    `json:"code"`
	Message string    `json:"message"`
	Action  ErrAction `json:"-"`
	Field   ErrField  `json:"error_field,omitempty"`
}

func (e ErrorMsg) Error() string {
	errorMsg, err := json.Marshal(e)
	if err != nil {
		return "unexpected failure"
	}
	return string(errorMsg)
}

type ErrorMsg struct {
	ErrResp `json:"error"`
	ErrType ErrType `json:"-"`
}

var ConnectInvalidToken = ErrorMsg{ErrResp{
	Code:    "invalid_connect_token",
	Message: "auth token is not valid",
}, AuthErr}

var DeltaInvalid = ErrorMsg{ErrResp{
	Code:    "invalid_delta",
	Message: "delta message is invalid, unable to merge",
}, BadInput}

var DeltaMetadataFieldImpaired = ErrorMsg{ErrResp{
	Code:    "delta_metadata_field_impaired",
	Message: "delta message not valid due to invalid metadata field changes",
}, BadInput}

var DeltaMaxSizeExceeded = ErrorMsg{ErrResp{
	Code:    "delta_max_size_exceeded",
	Message: "delta message exceeds 20KB",
}, BadInput}

var ConnectMaxSizeExceeded = ErrorMsg{ErrResp{
	Code:    "connect_max_size_exceeded",
	Message: "connect message exceeds 100KB",
}, BadInput}

var DeltaRateLimitExceeded = ErrorMsg{ErrResp{
	Code:    "delta_rate_limit_exceeded",
	Message: "delta message send rate is faster than 1 message per second",
	Action:  ActionDisconnect,
}, BadRequestRate}

var UnexpectedFailure = ErrorMsg{ErrResp{
	Code:    "unexpected_failure",
	Message: "unexpected failure during communication",
}, ServerErr}

var ConnectionNotActive = ErrorMsg{ErrResp{
	Code:    "connection_not_active",
	Message: "connection not actively sending messages at expected rate",
	Action:  ActionDisconnect,
}, BadRequestRate}

var ConnectClientNotWhitelisted = ErrorMsg{ErrResp{
	Code:    "connect_client_not_whitelisted",
	Message: "client is not whitelisted to publish data for the current game",
}, AuthErr}

var RefreshMissingInfo = ErrorMsg{ErrResp{
	Code:    "refresh_missing_info",
	Message: "refresh message is missing required field",
}, BadInput}

var ConnectMissingInfo = ErrorMsg{ErrResp{
	Code:    "connect_missing_info",
	Message: "connect message is missing required field",
}, BadInput}

var ConnectBadInfo = ErrorMsg{ErrResp{
	Code:    "connect_invalid_values",
	Message: "connect message has an invalid value for a required field",
}, BadInput}

var RefreshMaxSizeExceeded = ErrorMsg{ErrResp{
	Code:    "refresh_data_exceeding_limit",
	Message: "refresh message exceeds 100KB",
}, BadInput}

var BadDataPack = ErrorMsg{ErrResp{
	Code:    "invalid_message",
	Message: "message is not an accepted type or cannot be parsed in expected format",
}, BadInput}

var ConnectionNotAuthed = ErrorMsg{ErrResp{
	Code:    "connection_not_authed",
	Message: "connection is not authorized to send delta messages, please send connect message first",
}, AuthErr}

var NotWhitelistedAppToken = ErrorMsg{ErrResp{
	Code:    "not_whitelisted_app_token",
	Message: "client for this app token is not whitelisted to send messages",
}, AuthErr}

var ExceedingBufferSize = ErrorMsg{ErrResp{
	Code:    "exceeding_buffer_size",
	Message: "exceeded server buffer size, please send a refresh message",
	Action:  ActionReconnect,
}, ServerErr}

var ConnectReAuthNotSupported = ErrorMsg{ErrResp{
	Code:    "connect_reauth_not_supported",
	Message: "cannot send connect message again if already authed",
}, AuthErr}

var ServerFailure = ErrorMsg{ErrResp{
	Code:    "server_failure",
	Message: "server failure during communication",
}, ServerErr}

var WaitingOnRefreshPack = ErrorMsg{ErrResp{
	Code:    "waiting_on_refresh_message",
	Message: "error previously encountered, awaiting a refresh message to refresh data",
	Action:  ActionDoNothing,
}, BadInput}

func BuildErrMsgByField(errMsg ErrorMsg, field ErrField) *ErrorMsg {
	fieldErr := errMsg
	fieldErr.Field = field
	return &fieldErr
}
