package sns

import (
	"context"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"

	"code.justin.tv/devhub/twitch-e2-ingest/iam"
)

type TopicEnv string

const (
	TopicEnvDev  TopicEnv = "dev"
	TopicEnvProd TopicEnv = "prod"
)

type Topic struct {
	ARN  string
	Name string
}

const LEAGUE_GAME_ID = "21779"

// Creates an SNS topic for the game, client, and environment.
// This function is idempotent, so if the SNS topic already exists, the function is a noop.
func (p *publisherImpl) CreateTopic(ctx context.Context, gameID string, clientID string, env TopicEnv) (*Topic, error) {
	name := topicName(gameID, clientID, env)
	output, err := p.snsClient.CreateTopicWithContext(ctx, &sns.CreateTopicInput{
		Name: aws.String(name),
	})
	if err != nil {
		return nil, err
	}

	accountIDs := p.config.AWSAccountIDs
	principals := accountIDs.AllAccess
	if gameID == LEAGUE_GAME_ID {
		principals = append(principals, accountIDs.LeagueCME...)
	}

	policy, err := iam.PolicyDocumentJSON(iam.PolicyStatement{
		Action:    []string{"sns:Subscribe"},
		Condition: iam.StringEquals("sns:Protocol", "sqs"),
		Effect:    iam.EffectAllow,
		Principal: iam.AWSAccountIDs(principals...),
		Resource:  aws.StringValue(output.TopicArn),
	}, iam.PolicyStatement{
		Action:    []string{"sns:Subscribe", "sns:Publish"},
		Effect:    iam.EffectAllow,
		Principal: iam.AWSAccountIDs(accountIDs.E2IngestHTTP),
		Resource:  aws.StringValue(output.TopicArn),
	})
	if err != nil {
		return nil, err
	}

	_, err = p.snsClient.SetTopicAttributesWithContext(ctx, &sns.SetTopicAttributesInput{
		AttributeName:  aws.String("Policy"),
		AttributeValue: aws.String(policy),
		TopicArn:       output.TopicArn,
	})
	if err != nil {
		return nil, err
	}

	return &Topic{
		ARN:  *output.TopicArn,
		Name: name,
	}, nil
}

func topicName(gameID string, clientID string, env TopicEnv) string {
	return fmt.Sprintf("game-%s_client-%s_env-%s", gameID, clientID, env)
}
