package sns

import (
	"context"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/arn"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"golang.org/x/sync/errgroup"
)

var topicEnvs = []TopicEnv{TopicEnvDev, TopicEnvProd}

type Topics []*Topic

func (p *publisherImpl) CreateTopics(ctx context.Context, gameID string, clientID string) (Topics, error) {
	g := &errgroup.Group{}
	topics := make([]*Topic, len(topicEnvs))

	for i, env := range topicEnvs {
		i, env := i, env // https://golang.org/doc/faq#closures_and_goroutines
		g.Go(func() error {
			topic, err := p.CreateTopic(ctx, gameID, clientID, env)
			if err != nil {
				return err
			}

			topics[i] = topic
			return nil
		})
	}

	if err := g.Wait(); err != nil {
		return nil, err
	}

	return topics, nil
}

func (t Topics) MarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	av.SS = aws.StringSlice(t.ARNs())
	return nil
}

func (t *Topics) UnmarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	if av.SS == nil {
		return nil
	}
	arns := aws.StringValueSlice(av.SS)
	*t = topicsFromARNs(arns)
	return nil
}

func (topics Topics) ARNs() []string {
	arns := make([]string, 0, len(topics))
	for _, t := range topics {
		arns = append(arns, t.ARN)
	}
	return arns
}

func topicsFromARNs(arns []string) Topics {
	var topics Topics
	for _, a := range arns {
		var name string
		if p, err := arn.Parse(a); err == nil {
			name = p.Resource
		}
		topics = append(topics, &Topic{
			ARN:  a,
			Name: name,
		})
	}
	return topics
}
