package sns

import (
	"context"
	"errors"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	"code.justin.tv/devhub/twitch-e2-ingest/awsmocks/mocks"
	"code.justin.tv/devhub/twitch-e2-ingest/iam"
)

const (
	defaultTopicARN = "arn:aws:sns:us-east-2:797743463538:dev-mdaas-metadata"
)

type PublisherTest struct {
	suite.Suite
	publisher  Publisher
	mockSNSAPI *mocks.SNSAPI
}

func (suite *PublisherTest) SetupTest() {
	mSNS := &mocks.SNSAPI{}
	p, _ := NewTestPublisher(mSNS, &Config{
		DefaultTopicARN: defaultTopicARN,
		AWSRegion:       "us-west-2",
		AWSAccountIDs: AWSAccountIDs{
			E2IngestHTTP: "111",
			LeagueCME:    []string{"222"},
			AllAccess:    []string{"333", "444"},
		},
	})
	suite.publisher = p
	suite.mockSNSAPI = mSNS
}

func (suite *PublisherTest) TearDownTest() {
	suite.mockSNSAPI.AssertExpectations(suite.T())
}

func TestRunProcessorSuite(t *testing.T) {
	suite.Run(t, new(PublisherTest))
}

func (suite *PublisherTest) TestPublishSuccessTest() {
	message := map[string]string{
		"abc": "test1",
		"123": "test2",
	}

	suite.mockSNSAPI.On("Publish", &sns.PublishInput{
		Message:          aws.String("{\"default\":\"{\\\"123\\\":\\\"test2\\\",\\\"abc\\\":\\\"test1\\\"}\"}"),
		MessageStructure: aws.String("json"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(defaultTopicARN),
	}).Return(nil, nil)

	err := suite.publisher.Publish(message)
	suite.NoError(err)
}

func (suite *PublisherTest) TestPublishFailureTest() {
	message := map[string]string{
		"abc": "test1",
		"123": "test2",
	}

	suite.mockSNSAPI.On("Publish", &sns.PublishInput{
		Message:          aws.String("{\"default\":\"{\\\"123\\\":\\\"test2\\\",\\\"abc\\\":\\\"test1\\\"}\"}"),
		MessageStructure: aws.String("json"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(defaultTopicARN),
	}).Return(nil, errors.New("failed sns publish")).Times(10)

	err := suite.publisher.Publish(message)
	suite.Error(err)
	suite.Equal("failed sns publish", err.Error())

	time.Sleep(1 * time.Second)
}

func (suite *PublisherTest) TestPublishFailureThenSuccessTest() {
	message := map[string]string{
		"abc": "test1",
		"123": "test2",
	}

	suite.mockSNSAPI.On("Publish", &sns.PublishInput{
		Message:          aws.String("{\"default\":\"{\\\"123\\\":\\\"test2\\\",\\\"abc\\\":\\\"test1\\\"}\"}"),
		MessageStructure: aws.String("json"),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(defaultTopicARN),
	}).Return(nil, errors.New("failed sns publish")).
		Times(4).
		Return(nil, nil).
		Times(1)

	err := suite.publisher.Publish(message)
	suite.NoError(err)

	time.Sleep(1 * time.Second)
}

func (suite *PublisherTest) TestCreateTopicSuccess() {
	topicARN := "arn:aws:sns:us-west-2:797743463538:game-testGame_client-testClient_env-dev"
	suite.mockSNSAPI.On("CreateTopicWithContext", mock.Anything, &sns.CreateTopicInput{
		Name: aws.String("game-testGame_client-testClient_env-dev"),
	}).Return(&sns.CreateTopicOutput{TopicArn: aws.String(topicARN)}, nil)

	policy, err := iam.PolicyDocumentJSON(iam.PolicyStatement{
		Action:    []string{"sns:Subscribe"},
		Condition: iam.StringEquals("sns:Protocol", "sqs"),
		Effect:    iam.EffectAllow,
		Principal: iam.AWSAccountIDs([]string{"333", "444"}...),
		Resource:  topicARN,
	}, iam.PolicyStatement{
		Action:    []string{"sns:Subscribe", "sns:Publish"},
		Effect:    iam.EffectAllow,
		Principal: iam.AWSAccountIDs("111"),
		Resource:  topicARN,
	})
	suite.mockSNSAPI.On("SetTopicAttributesWithContext", mock.Anything, &sns.SetTopicAttributesInput{
		AttributeName:  aws.String("Policy"),
		AttributeValue: aws.String(policy),
		TopicArn:       aws.String(topicARN),
	}).Return(&sns.SetTopicAttributesOutput{}, nil)
	suite.NoError(err)

	topic, err := suite.publisher.CreateTopic(context.Background(), "testGame", "testClient", TopicEnvDev)
	suite.NoError(err)
	suite.Equal(topic, &Topic{
		ARN:  topicARN,
		Name: "game-testGame_client-testClient_env-dev",
	})
}

func (suite *PublisherTest) TestCreateTopicLeagueSuccess() {
	topicARN := "arn:aws:sns:us-west-2:797743463538:game-21779_client-testClient_env-dev"
	suite.mockSNSAPI.On("CreateTopicWithContext", mock.Anything, &sns.CreateTopicInput{
		Name: aws.String("game-21779_client-testClient_env-dev"),
	}).Return(&sns.CreateTopicOutput{TopicArn: aws.String(topicARN)}, nil)

	policy, err := iam.PolicyDocumentJSON(iam.PolicyStatement{
		Action:    []string{"sns:Subscribe"},
		Condition: iam.StringEquals("sns:Protocol", "sqs"),
		Effect:    iam.EffectAllow,
		Principal: iam.AWSAccountIDs([]string{"333", "444", "222"}...),
		Resource:  topicARN,
	}, iam.PolicyStatement{
		Action:    []string{"sns:Subscribe", "sns:Publish"},
		Effect:    iam.EffectAllow,
		Principal: iam.AWSAccountIDs("111"),
		Resource:  topicARN,
	})
	suite.mockSNSAPI.On("SetTopicAttributesWithContext", mock.Anything, &sns.SetTopicAttributesInput{
		AttributeName:  aws.String("Policy"),
		AttributeValue: aws.String(policy),
		TopicArn:       aws.String(topicARN),
	}).Return(&sns.SetTopicAttributesOutput{}, nil)
	suite.NoError(err)

	topic, err := suite.publisher.CreateTopic(context.Background(), "21779", "testClient", TopicEnvDev)
	suite.NoError(err)
	suite.Equal(topic, &Topic{
		ARN:  topicARN,
		Name: "game-21779_client-testClient_env-dev",
	})
}

func (suite *PublisherTest) TestCreateTopicCreateFailure() {
	suite.mockSNSAPI.On("CreateTopicWithContext", mock.Anything, mock.Anything).Return(nil, errors.New("oops"))

	topic, err := suite.publisher.CreateTopic(context.Background(), "testGame", "testClient", TopicEnvDev)
	suite.Error(err)
	suite.Nil(topic)

	suite.mockSNSAPI.AssertNotCalled(suite.T(), "SetTopicAttributesWithContext")
}

func (suite *PublisherTest) TestCreateTopicPolicyFailure() {
	suite.mockSNSAPI.On("CreateTopicWithContext", mock.Anything, mock.Anything).
		Return(&sns.CreateTopicOutput{TopicArn: aws.String("topicARN")}, nil)
	suite.mockSNSAPI.On("SetTopicAttributesWithContext", mock.Anything, mock.Anything).Return(nil, errors.New("oops"))

	topic, err := suite.publisher.CreateTopic(context.Background(), "testGame", "testClient", TopicEnvDev)
	suite.Error(err)
	suite.Nil(topic)
}

func (suite *PublisherTest) TestCreateTopicsSuccess() {
	topicARNs := map[string]string{
		"arn:aws:sns:us-west-2:797743463538:game-testGame_client-testClient_env-dev":  "game-testGame_client-testClient_env-dev",
		"arn:aws:sns:us-west-2:797743463538:game-testGame_client-testClient_env-prod": "game-testGame_client-testClient_env-prod",
	}
	for arn, name := range topicARNs {
		suite.mockSNSAPI.On("CreateTopicWithContext", mock.Anything, &sns.CreateTopicInput{Name: aws.String(name)}).
			Return(&sns.CreateTopicOutput{TopicArn: aws.String(arn)}, nil)
	}

	suite.mockSNSAPI.On("SetTopicAttributesWithContext", mock.Anything, mock.Anything).Return(&sns.SetTopicAttributesOutput{}, nil)

	topics, err := suite.publisher.CreateTopics(context.Background(), "testGame", "testClient")
	suite.NoError(err)
	suite.Len(topics, 2)

	for _, topic := range topics {
		name, ok := topicARNs[topic.ARN]
		suite.True(ok)
		suite.Equal(topic.Name, name)
		delete(topicARNs, topic.ARN)
	}
	suite.Empty(topicARNs)

	suite.mockSNSAPI.AssertNumberOfCalls(suite.T(), "CreateTopicWithContext", 2)
	suite.mockSNSAPI.AssertNumberOfCalls(suite.T(), "SetTopicAttributesWithContext", 2)
}

func (suite *PublisherTest) TestCreateTopicsFailure() {
	suite.mockSNSAPI.On("CreateTopicWithContext", mock.Anything, mock.Anything).
		Return(&sns.CreateTopicOutput{TopicArn: aws.String("topicARN")}, nil).Times(1)
	suite.mockSNSAPI.On("CreateTopicWithContext", mock.Anything, mock.Anything).
		Return(nil, errors.New("oops")).Times(1)
	suite.mockSNSAPI.On("SetTopicAttributesWithContext", mock.Anything, mock.Anything).Return(&sns.SetTopicAttributesOutput{}, nil).Times(1)

	topics, err := suite.publisher.CreateTopics(context.Background(), "testGame", "testClient")
	suite.Error(err)
	suite.Nil(topics)

	suite.mockSNSAPI.AssertNumberOfCalls(suite.T(), "CreateTopicWithContext", 2)
	suite.mockSNSAPI.AssertNumberOfCalls(suite.T(), "SetTopicAttributesWithContext", 1)
}
