from io import BytesIO
import boto3
import gspread
from collections import defaultdict
import traceback

def build_query_map(query_sheet,connection):
    """ Takes in the MASTER SQL worksheet and builds out a mapping of query_id to query text and a psycopg2 cursor that contains the results. """

    #Building out the map with the query_id as the key.
    query_map = defaultdict(tuple)
    values = query_sheet.get_all_values()

    #Iterating through all the rows except the first row as the first row are headers.
    for row_index in range(len(values)):
        if row_index == 0:
            continue
        else:
            #print(values[row_index])
            # Attempts to execute the query. If there's an error, the cursor is replaced with a string containing the error message.
            try:
                cursor = connection.cursor()
                cursor.execute(values[row_index][1])
                query_map[values[row_index][0]] = (values[row_index][1],cursor)
                connection.commit()
            except:
                query_map[values[row_index][0]] = (values[row_index][1],traceback.format_exc())

    #Converting into a dictionary since it's harder to modify a dictionary.
    return dict(query_map)

def upload(query_sheet,connection):
    """ Writes the results of each query into a csv and is uploaded onto S3. """

    print('Building map')
    # Build out the map with the query_id as the key.
    query_map = build_query_map(query_sheet, connection)

    # Create the S3 Client to upload the csv into.
    s3 = boto3.client('s3')

    # Iterating through each query_id.
    for query_id in query_map.keys():
        # We're using BytesIO since we're able to create a buffer so that we don't need to save a local file on coconut_box.
        byte_file = BytesIO()

        # Check for whether the command threw an error or not
        if type(query_map[query_id][1]) == str:
            byte_file.write('{}\n{}'.format(query_id,'ERROR').encode('utf-8'))
        else:
            # If there's no error, write all of the rows fetched into the byte_file.
            cursor = query_map[query_id][1]
            columns = [column.name for column in cursor.description]
            byte_file.write('{}\n{}\n'.format(query_id,','.join(columns)).encode('utf-8'))
            while True:
                # Fetch 5000 rows at a time.
                rows = cursor.fetchmany(5000)
                if not rows:
                    break
                for row in rows:
                    row = [str(element) for element in row]
                    byte_file.write('{}\n'.format(','.join(row)).encode('utf-8'))
        # Get the buffer at the correct position.
        byte_file.seek(0)

        # Write the buffer into S3 and close the buffer.
        s3.put_object(Bucket='query-to-googlesheet',Key = '{}.csv'.format(query_id),Body = byte_file)
        byte_file.close()

