package main

import (
	"fmt"
	"log"
	"net"
	"net/http"
	"os"

	_ "net/http/pprof"

	"code.justin.tv/ear/kinesis/multilangclient"
)

func init() {
	// Run default HTTP server to expose profiling
	go func() {
		listener, err := net.Listen("tcp", "localhost:0")
		if err != nil {
			log.Fatalf("tcp listen err=%q", err)
		}

		log.Printf("listening on %s", listener.Addr())

		err = http.Serve(listener, nil)
		if err != nil {
			log.Fatalf("http serve err=%q", err)
		}
	}()
}

// messageLogger implements the kcl.Consumer interface by treating
// messages as strings and logging them to a file named based on the
// shard.
type messageLogger struct {
	f *os.File
	l *log.Logger
}

// Initialize creates a new log file for writing based on the shard
// ID.
func (m *messageLogger) Initialize(shardID string) error {
	var err error
	m.f, err = os.Create(fmt.Sprintf("messagelogs.%s.log", shardID))
	if err != nil {
		return err
	}
	m.l = log.New(m.f, fmt.Sprintf("[messagelogger-%s] ", shardID), log.LstdFlags)
	return nil
}

// Process converts the record to a string and writes it the
// messageLoggers' log file.
func (m *messageLogger) Process(record []byte) error {
	m.l.Println(string(record))
	return nil
}

// Shutdown closes the messageLogger's file handle.
func (m *messageLogger) Shutdown() error {
	return m.f.Close()
}

func main() {
	c := &messageLogger{}
	err := multilangclient.Run(c)
	if err != nil {
		log.Fatalf("err=%v", err)
	}
}
