import * as exp from 'expect';
import { identifyDevice } from '../src/device-identifier';

const expect = exp as any as Function;

describe('identifyDevice', () => {
  describe('iOS devices', () => {
    const UA_STRINGS = [
      // iPad section
      'Mozilla/5.0 (iPad; CPU OS 9_3_4 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) GSA/17.0.128207670 Mobile/13G35 Safari/600.1.4',
      'Mozilla/5.0 (iPad; CPU iPad OS 8_4 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) Version/8.0 Mobile/12H143 Safari/600.1.4',
      // iPhone section
      'Mozilla/5.0 (iPhone; U; CPU iPhone OS 7_1_1 like Mac OS X; es-MX) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/30.0.1599.114 Safari/537.36 Puffin/3.7.0IP Mobile',
      'Mozilla/5.0 (iPhone; U; CPU like Mac OS X; en) AppleWebKit/420+ (KHTML, like Gecko) Version/3.0 Mobile/1C25 Safari/419.3',
      'Mozilla/5.0 (iPhone; U; ru; CPU iPhone OS 4_2_1 like Mac OS X; ru) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2 Mobile/8C148a Safari/6533.18.5',
      // iPod section
      'Mozilla/5.0 (iPod touch; CPU iPhone OS 7_0_3 like Mac OS X) AppleWebKit/537.51.1 (KHTML, like Gecko) Version/7.0 Mobile/11B511 Safari/9537.53',
      'Mozilla/5.0 (iPod; CPU iPhone OS 5_0_1 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Version/5.1 Mobile/9A405 Safari/7534.48.3',
      // Chrome on iOS example
      'Mozilla/5.0 (iPhone; U; CPU iPhone OS 5_1_1 like Mac OS X; en) AppleWebKit/534.46.0 (KHTML, like Gecko) CriOS/19.0.1084.60 Mobile/9B206 Safari/7534.48.3',
    ];

    UA_STRINGS.forEach((ua: string) => {
      expectItToIdentify(ua, true, false, false);
    });
  });

  describe('Android devices', () => {
    const UA_STRINGS = [
      // phone
      'Mozilla/5.0 (Linux; Android 4.0.4; Galaxy Nexus Build/IMM76B) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.133 Mobile Safari/535.19',
      // tablet
      'Mozilla/5.0 (Linux; Android 4.0.4; Galaxy Nexus Build/IMM76B) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.133 Safari/535.19',
      // Old webview
      'Mozilla/5.0 (Linux; U; Android 4.1.1; en-gb; Build/KLP) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Safari/534.30',
      // webview on KitKat & Lollipop
      'Mozilla/5.0 (Linux; Android 4.4; Nexus 5 Build/_BuildID_) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/30.0.0.0 Mobile Safari/537.36',
      // webview on lollipop & above
      'Mozilla/5.0 (Linux; Android 5.1.1; Nexus 5 Build/LMY48B; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/43.0.2357.65 Mobile Safari/537.36',
      // Firefox on Android section
      'Mozilla/5.0 (Android; Mobile; rv:40.0) Gecko/40.0 Firefox/40.0',
      'Mozilla/5.0 (Android; Tablet; rv:40.0) Gecko/40.0 Firefox/40.0',
      'Mozilla/5.0 (Android 4.4; Mobile; rv:41.0) Gecko/41.0 Firefox/41.0',
      'Mozilla/5.0 (Android 4.4; Tablet; rv:41.0) Gecko/41.0 Firefox/41.0'
    ];

    UA_STRINGS.forEach((ua: string) => {
      expectItToIdentify(ua, false, true, false);
    });
  });
  describe('Windows devices', () => {
    const UA_STRINGS = [
      // Old windows phone
      'Mozilla/5.0 (compatible; MSIE 10.0; Windows Phone 8.0; Trident/6.0; IEMobile/10.0; ARM; Touch; HTC; Windows Phone 8X by HTC)',
      'Mozilla/5.0 (compatible; MSIE 10.0; Windows Phone 8.0; Trident/6.0; IEMobile/10.0; ARM; Touch; NOKIA; Lumia 920)',
      // Internet Explorer for Windows Phone 8.1 Update
      'Mozilla/5.0 (Mobile; Windows Phone 8.1; Android 4.0; ARM; Trident/7.0; Touch; rv:11.0; IEMobile/11.0; NOKIA; Lumia 520) like iPhone OS 7_0_3 Mac OS X AppleWebKit/537 (KHTML, like Gecko) Mobile Safari/537',
      // New Edge browser with shifty UA string.
      'Mozilla/5.0 (Windows Phone 10.0; Android 4.2.1; Microsoft; Lumia 640 LTE) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/46.0.2486.0 Mobile Safari/537.36 Edge/13.10586',
    ];

    UA_STRINGS.forEach((ua: string) => {
      expectItToIdentify(ua, false, false, true);
    });
  });
  describe('Non-mobile devices', () => {
    const UA_STRINGS = [
      // Edge
      'Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.6 (KHTML, like Gecko) Chrome/42.0.0.1 Safari/538 Edge/13.111',
      // IE 11 on Windows 10
      'Mozilla/5.0 (Windows NT 10.0; WOW64; Trident/7.0; rv:11.0) like Gecko',
      // IE11 for the desktop on 64-bit Windows 8.1 Update
      'Mozilla/5.0 (Windows NT 6.3; WOW64; Trident/7.0; Touch; rv:11.0) like Gecko',
      // IE11 for the desktop on 64-bit Windows 8.1 Update with enterprise mode enabled
      'Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.1; Trident/4.0; Tablet PC 2.0)',
      // Safari
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_2) AppleWebKit/600.3.18 (KHTML, like Gecko) Version/8.0.3 Safari/600.3.18',
      // Chrome on OS X
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/52.0.2743.82 Safari/537.36',
      // Firefox on OS X
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.11; rv:45.0) Gecko/20100101 Firefox/45.0',
      // IE on XBone
      'Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.2; Trident/6.0; Xbox; Xbox One)',
    ];

    UA_STRINGS.forEach((ua: string) => {
      expectItToIdentify(ua, false, false, false, false);
    });
  })
});

function expectItToIdentify(
  ua: string,
  isIOS: boolean,
  isAndroid: boolean,
  isWindows: boolean,
  isMobile = true
): void {
  it(`recognizes whether "${ua}" is for iOS`, () => {
    expect(identifyDevice(ua).isIOS()).toEqual(isIOS);
  });

  it(`recognizes whether "${ua}" is for Android`, () => {
    expect(identifyDevice(ua).isAndroid()).toEqual(isAndroid);
  });

  it(`recognizes whether "${ua}" is for Windows`, () => {
    expect(identifyDevice(ua).isWindows()).toEqual(isWindows);
  });

  it(`recognizes whether "${ua} is a mobile device`, () => {
    expect(identifyDevice(ua).isMobile()).toEqual(isMobile);
  });
}