variable "resource_id" {}

variable "resource_rest_api_id" {}

variable "lambda_region" {}

variable "lambda_name" {}

variable "role_arn" {}

variable "request_parameters" {
  type = "map"
}

variable "success_integration_response_parameters" {
  type = "map"
}

variable "success_response_templates" {
  type = "map"
}

variable "success_response_parameters" {
  type = "map"
}

variable "integration_request_templates" {
  type = "map"
}

variable "method" {
  type    = "string"
  default = "GET"
}

# GET method for resource
resource "aws_api_gateway_method" "method" {
  rest_api_id        = "${var.resource_rest_api_id}"
  resource_id        = "${var.resource_id}"
  http_method        = "${var.method}"
  authorization      = "NONE"
  request_parameters = "${var.request_parameters}"
}

resource "aws_api_gateway_integration" "method_integration" {
  rest_api_id             = "${var.resource_rest_api_id}"
  resource_id             = "${var.resource_id}"
  http_method             = "${aws_api_gateway_method.method.http_method}"
  integration_http_method = "POST"
  type                    = "AWS"
  uri                     = "arn:aws:apigateway:${var.lambda_region}:lambda:path/2015-03-31/functions/${var.lambda_name}/invocations"
  credentials             = "${var.role_arn}"
  request_templates       = "${var.integration_request_templates}"
}

# 200 response integration response
resource "aws_api_gateway_integration_response" "success_integration_response" {
  depends_on = [
    "aws_api_gateway_integration.method_integration",
  ]

  rest_api_id = "${var.resource_rest_api_id}"
  resource_id = "${var.resource_id}"
  http_method = "${aws_api_gateway_method.method.http_method}"
  status_code = "${aws_api_gateway_method_response.success_response.status_code}"

  # Success is specifically an empty error string.
  selection_pattern   = "^$"
  response_parameters = "${var.success_integration_response_parameters}"
  response_templates  = "${var.success_response_templates}"
}

# Method 200 Response
resource "aws_api_gateway_method_response" "success_response" {
  rest_api_id = "${var.resource_rest_api_id}"
  resource_id = "${var.resource_id}"

  http_method = "${aws_api_gateway_method.method.http_method}"
  status_code = "200"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = "${var.success_response_parameters}"
}

# 500 response integration
resource "aws_api_gateway_integration_response" "internal_server_error_integration_response" {
  rest_api_id       = "${var.resource_rest_api_id}"
  resource_id       = "${var.resource_id}"
  http_method       = "${aws_api_gateway_method.method.http_method}"
  status_code       = "${aws_api_gateway_method_response.internal_server_error_response.status_code}"
  selection_pattern = ""

  response_templates = {
    "application/json" = "#set ($errorMessageObj = $util.parseJson($input.path('$.errorMessage')))\n$errorMessageObj.html"
  }

  response_parameters = {
    "method.response.header.Content-Type" = "'text/html; charset=utf-8'"
  }
}

# Method 500 Response
resource "aws_api_gateway_method_response" "internal_server_error_response" {
  depends_on = [
    "aws_api_gateway_integration.method_integration",
  ]

  rest_api_id = "${var.resource_rest_api_id}"
  resource_id = "${var.resource_id}"

  http_method = "${aws_api_gateway_method.method.http_method}"
  status_code = "500"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = {
    "method.response.header.Content-Type" = true
  }
}

# 504 response integration
resource "aws_api_gateway_integration_response" "gateway_timeout_integration_response" {
  depends_on = [
    "aws_api_gateway_integration.method_integration",
  ]

  rest_api_id       = "${var.resource_rest_api_id}"
  resource_id       = "${var.resource_id}"
  http_method       = "${aws_api_gateway_method.method.http_method}"
  status_code       = "${aws_api_gateway_method_response.gateway_timeout_response.status_code}"
  selection_pattern = "Task timed out after"
}

# Method 504 Response
resource "aws_api_gateway_method_response" "gateway_timeout_response" {
  rest_api_id = "${var.resource_rest_api_id}"
  resource_id = "${var.resource_id}"

  http_method = "${aws_api_gateway_method.method.http_method}"
  status_code = "504"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = {
    "method.response.header.Content-Type" = true
  }
}

# 404 response integration
resource "aws_api_gateway_integration_response" "not_found_integration_response" {
  depends_on = [
    "aws_api_gateway_integration.method_integration",
  ]

  rest_api_id       = "${var.resource_rest_api_id}"
  resource_id       = "${var.resource_id}"
  http_method       = "${aws_api_gateway_method.method.http_method}"
  status_code       = "${aws_api_gateway_method_response.not_found_response.status_code}"
  selection_pattern = ".*\\\"status\\\":\\\"not found\\\".*"

  response_templates = {
    "application/json" = "#set ($errorMessageObj = $util.parseJson($input.path('$.errorMessage')))\n$errorMessageObj.html"
  }

  response_parameters = {
    "method.response.header.Content-Type" = "'text/html; charset=utf-8'"
  }
}

# Method 404 Response
resource "aws_api_gateway_method_response" "not_found_response" {
  rest_api_id = "${var.resource_rest_api_id}"
  resource_id = "${var.resource_id}"

  http_method = "${aws_api_gateway_method.method.http_method}"
  status_code = "404"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = {
    "method.response.header.Content-Type" = true
  }
}

# 302 response integration
resource "aws_api_gateway_integration_response" "redirect_integration_response" {
  depends_on = [
    "aws_api_gateway_integration.method_integration",
  ]

  rest_api_id       = "${var.resource_rest_api_id}"
  resource_id       = "${var.resource_id}"
  http_method       = "${aws_api_gateway_method.method.http_method}"
  status_code       = "${aws_api_gateway_method_response.redirect_response.status_code}"
  selection_pattern = "^https://www.twitch.tv.*"

  response_templates = {
    "application/json" = "empty"
  }

  response_parameters = {
    "method.response.header.Location" = "integration.response.body.errorMessage"
  }
}

# Method 302 Response
resource "aws_api_gateway_method_response" "redirect_response" {
  rest_api_id = "${var.resource_rest_api_id}"
  resource_id = "${var.resource_id}"

  http_method = "${aws_api_gateway_method.method.http_method}"
  status_code = "302"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = {
    "method.response.header.Location" = true
  }
}

output "method" {
  value = "${aws_api_gateway_integration.method_integration.http_method}"
}
