variable "aws_region" {}

variable "api_stage" {}

variable "gateway_invoke_lambda_role_arn" {}

variable "lambda_name" {}

variable "gateway_name" {
  default = "MWS Gateway"
}

variable basic_success_response_parameters {
  type = "map"

  default = {
    "method.response.header.Content-Type" = true
    "method.response.header.Cache-Control" = true
  }
}

variable basic_integration_response_parameters {
  type = "map"

  default = {
    "method.response.header.Content-Type" = "'text/html; charset=utf-8'"
    "method.response.header.Cache-Control" = "integration.response.body.headers.cacheControl"
  }
}

variable basic_success_response_templates {
  type = "map"

  default = {
    "application/json" = "$input.path('$').html"
  }
}

resource "aws_api_gateway_rest_api" "mws_gateway" {
  description = "API Gateway for mobile web service"
  name        = "${var.gateway_name}"
}

module "landing_page_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_rest_api.mws_gateway.root_resource_id}"
  resource_rest_api_id                    = "${aws_api_gateway_rest_api.mws_gateway.id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "directory_main_game",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('Desktop-Redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": []
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /directory
resource "aws_api_gateway_resource" "games_directory" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_rest_api.mws_gateway.root_resource_id}"
  path_part   = "directory"
}

module "games_directory_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.games_directory.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.games_directory.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "directory_main_game",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('Desktop-Redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["directory"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /directory/game
resource "aws_api_gateway_resource" "game_endpoint" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.games_directory.id}"
  path_part   = "game"
}

# Resource /directory/game/{game}
resource "aws_api_gateway_resource" "game_directory" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.game_endpoint.id}"
  path_part   = "{game}"
}

module "game_directory_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.game_directory.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.game_directory.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.game"                        = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "directory_game",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "game": "$input.params('game')",
  "pathParts": ["directory", "game", "$input.params('game')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /directory/all
resource "aws_api_gateway_resource" "channels_directory" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.games_directory.id}"
  path_part   = "all"
}

module "channels_directory_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channels_directory.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channels_directory.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "directory_game",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "game": "CHANNEL_DIRECTORY_FOR_ALL_GAMES",
  "pathParts": ["directory", "all"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /videos
resource "aws_api_gateway_resource" "video_endpoint" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_rest_api.mws_gateway.root_resource_id}"
  path_part   = "videos"
}

# Resource /videos/{id}
resource "aws_api_gateway_resource" "video_viewer" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.video_endpoint.id}"
  path_part   = "{id}"
}

module "video_viewer_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.video_viewer.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.video_viewer.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.id"                          = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "vod",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "id": "$input.params('id')",
  "pathParts": ["videos", "$input.params('id')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /events
resource "aws_api_gateway_resource" "event_prefix" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_rest_api.mws_gateway.root_resource_id}"
  path_part   = "events"
}

# Resource /events/{event_id}
resource "aws_api_gateway_resource" "event_details" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.event_prefix.id}"
  path_part   = "{event_id}"
}

module "event_details_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.event_details.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.event_details.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.event_id"                    = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "event_details",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "eventID": "$input.params('event_id')",
  "pathParts": ["events", "$input.params('event_id')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}
resource "aws_api_gateway_resource" "channel_viewer" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_rest_api.mws_gateway.root_resource_id}"
  path_part   = "{channel}"
}

module "channel_viewer_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_viewer.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_viewer.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "channel": "$input.params('channel')",
  "pathParts": ["$input.params('channel')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/rochat
resource "aws_api_gateway_resource" "channel_readonly_chat" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "rochat"
}

module "channel_chat_embed_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_readonly_chat.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_readonly_chat.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "chat_embed",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "rochat"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/profile
resource "aws_api_gateway_resource" "channel_profile" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "profile"
}

module "channel_profile_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_profile.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_profile.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
    "method.request.querystring.desktop-redirect"     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "profile"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/clips
resource "aws_api_gateway_resource" "channel_clips" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "clips"
}

module "channel_clips_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_clips.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_clips.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "clips"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/collections
resource "aws_api_gateway_resource" "channel_collections" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "collections"
}

module "channel_collections_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_collections.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_collections.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "collections"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/events
resource "aws_api_gateway_resource" "channel_events" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "events"
}

module "channel_events_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_events.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_events.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "events"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/followers
resource "aws_api_gateway_resource" "channel_followers" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "followers"
}

module "channel_followers_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_followers.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_followers.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "followers"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/following
resource "aws_api_gateway_resource" "channel_following" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "following"
}

module "channel_following_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_following.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_following.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "following"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/videos
resource "aws_api_gateway_resource" "channel_videos" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "videos"
}

module "channel_videos_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_videos.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_videos.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.theme"                = true
    "method.request.querystring.fontsize"             = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "theme": "$input.params('theme')",
  "fontSize": "$input.params('fontsize')",
  "channel": "$input.params('channel')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "pathParts": ["$input.params('channel')", "videos"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/videos/{category}
resource "aws_api_gateway_resource" "channel_videos_by_category" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_videos.id}"
  path_part   = "{category}"
}

module "channel_videos_category_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_videos_by_category.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_videos_by_category.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
    "method.request.path.category"                    = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "channel_profile",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "channel": "$input.params('channel')",
  "id": "$input.params('id')",
  "pathParts": ["$input.params('channel')", "videos", "$input.params('category')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/v
resource "aws_api_gateway_resource" "channel_video_endpoint" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "v"
}

# Resource /{channel}/v/{vod_id}
resource "aws_api_gateway_resource" "channel_video_viewer" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_video_endpoint.id}"
  path_part   = "{vod_id}"
}

module "channel_video_viewer_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_video_viewer.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_video_viewer.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
    "method.request.path.id"                          = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "vod",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "channel": "$input.params('channel')",
  "id": "$input.params('id')",
  "pathParts": ["$input.params('channel')", "v", "$input.params('vod_id')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/{whatever}
resource "aws_api_gateway_resource" "channel_upsell_level_2" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_viewer.id}"
  path_part   = "{whatever}"
}

module "channel_upsell_level_2_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_upsell_level_2.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_upsell_level_2.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
    "method.request.path.whatever"                    = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "mobile_upsell",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "channel": "$input.params('channel')",
  "whatever": "$input.params('whatever')",
  "pathParts": ["$input.params('channel')", "$input.params('whatever')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /{channel}/{whatever}/{thirdThing}
resource "aws_api_gateway_resource" "channel_upsell_level_3" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_upsell_level_2.id}"
  path_part   = "{thirdThing}"
}

module "channel_upsell_level_3_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.channel_upsell_level_3.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.channel_upsell_level_3.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
    "method.request.path.whatever"                    = true
    "method.request.path.thirdThing"                  = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "mobile_upsell",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "channel": "$input.params('channel')",
  "whatever": "$input.params('whatever')",
  "thirdThing": "$input.params('thirdThing')",
  "pathParts": ["$input.params('channel')", "$input.params('whatever')", "$input.params('thirdThing')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Resource /streams
resource "aws_api_gateway_resource" "streams" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_rest_api.mws_gateway.root_resource_id}"
  path_part   = "streams"
}

# Resource /streams/<stream_id>
resource "aws_api_gateway_resource" "streams_stream_id" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.streams.id}"
  path_part   = "{stream_id}"
}

# Resource /streams/<stream_id>/channel
resource "aws_api_gateway_resource" "streams_channel" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.streams_stream_id.id}"
  path_part   = "channel"
}

# Resource /streams/<stream_id>/channel/<channel_id>
resource "aws_api_gateway_resource" "streams_channel_id" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.streams_channel.id}"
  path_part   = "{channel_id}"
}

module "streams_channel_id_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.streams_channel_id.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.streams_channel_id.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel_id"                  = true
    "method.request.path.stream_id"                   = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "live_to_vod",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "channel_id": "$input.params('channel_id')",
  "stream_id": "$input.params('stream_id')",
  "pathParts": ["streams", "$input.params('stream_id')", "channel", "$input.params('channel_id')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

# Custom 308 response for /videos/{id} on streams_channel_id
resource "aws_api_gateway_integration_response" "redirect_for_ltv_to_video_integration_response" {
  rest_api_id       = "${aws_api_gateway_rest_api.mws_gateway.id}"
  resource_id       = "${aws_api_gateway_resource.streams_channel_id.id}"
  http_method       = "${module.streams_channel_id_endpoint.method}"
  status_code       = "${aws_api_gateway_method_response.live_to_channel_permanent_redirect_response.status_code}"
  selection_pattern = "^/videos/.*"

  response_templates = {
    "application/json" = "empty"
  }

  response_parameters = {
    "method.response.header.Location"      = "integration.response.body.errorMessage"
    "method.response.header.Cache-Control" = "'public, max-age=1800'"
  }
}

resource "aws_api_gateway_method_response" "live_to_channel_permanent_redirect_response" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  resource_id = "${aws_api_gateway_resource.streams_channel_id.id}"

  http_method = "${module.streams_channel_id_endpoint.method}"
  status_code = "308"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = {
    "method.response.header.Location"      = true
    "method.response.header.Cache-Control" = true
  }
}

# Custom 307 response for /{channel} on streams_channel_id
resource "aws_api_gateway_integration_response" "redirect_for_ltv_to_channel_integration_response" {
  rest_api_id       = "${aws_api_gateway_rest_api.mws_gateway.id}"
  resource_id       = "${aws_api_gateway_resource.streams_channel_id.id}"
  http_method       = "${module.streams_channel_id_endpoint.method}"
  status_code       = "${aws_api_gateway_method_response.live_to_channel_temporary_redirect_response.status_code}"
  selection_pattern = "^/[^/]*$"

  response_templates = {
    "application/json" = "empty"
  }

  response_parameters = {
    "method.response.header.Location"      = "integration.response.body.errorMessage"
    "method.response.header.Cache-Control" = "'public, max-age=60'"
  }
}

resource "aws_api_gateway_method_response" "live_to_channel_temporary_redirect_response" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  resource_id = "${aws_api_gateway_resource.streams_channel_id.id}"

  http_method = "${module.streams_channel_id_endpoint.method}"
  status_code = "307"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = {
    "method.response.header.Location"      = true
    "method.response.header.Cache-Control" = true
  }
}

# Custom 307 response for /{channel}/profile from /{channel}
resource "aws_api_gateway_integration_response" "redirect_offline_channel_to_profile_integration_response" {
  rest_api_id       = "${aws_api_gateway_rest_api.mws_gateway.id}"
  resource_id       = "${aws_api_gateway_resource.channel_viewer.id}"
  http_method       = "${module.channel_viewer_endpoint.method}"
  status_code       = "${aws_api_gateway_method_response.redirect_offline_channel_to_profile_method_response.status_code}"
  selection_pattern = "^/[^/]*/profile?.*"

  response_templates = {
    "application/json" = "empty"
  }

  response_parameters = {
    "method.response.header.Location"      = "integration.response.body.errorMessage"
    "method.response.header.Cache-Control" = "'public, max-age=60'"
  }
}

resource "aws_api_gateway_method_response" "redirect_offline_channel_to_profile_method_response" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  resource_id = "${aws_api_gateway_resource.channel_viewer.id}"

  http_method = "${module.channel_viewer_endpoint.method}"
  status_code = "307"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = {
    "method.response.header.Location"      = true
    "method.response.header.Cache-Control" = true
  }
}

# 404
resource "aws_api_gateway_resource" "catchall_resource" {
  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  parent_id   = "${aws_api_gateway_resource.channel_upsell_level_3.id}"
  path_part   = "{theRest+}"
}

module "catchall_resource_endpoint" {
  source                                  = "../endpoint"
  resource_id                             = "${aws_api_gateway_resource.catchall_resource.id}"
  resource_rest_api_id                    = "${aws_api_gateway_resource.catchall_resource.rest_api_id}"
  lambda_region                           = "${var.aws_region}"
  lambda_name                             = "${var.lambda_name}"
  role_arn                                = "${var.gateway_invoke_lambda_role_arn}"
  success_integration_response_parameters = "${var.basic_integration_response_parameters}"
  success_response_parameters             = "${var.basic_success_response_parameters}"
  success_response_templates              = "${var.basic_success_response_templates}"

  request_parameters = {
    "method.request.header.Cookie"                    = true
    "method.request.header.Accept-Language"           = true
    "method.request.header.Experiment-Bucket"         = true
    "method.request.querystring.desktop-redirect"     = true
    "method.request.querystring.tt_medium"            = true
    "method.request.querystring.tt_content"           = true
    "method.request.querystring.experiment_overrides" = true
    "method.request.path.channel"                     = true
    "method.request.path.whatever"                    = true
    "method.request.path.thirdThing"                  = true
    "method.request.path.thirdThing"                  = true
    "method.request.path.theRest"                     = true
  }

  integration_request_templates = {
    "application/json" = <<END_OF_INTEGRATION_TEMPLATE
{
  "location": "not_found",
  "cookie": "$input.params('Cookie')",
  "language": "$input.params('Accept-Language')",
  "bucket_id": "$input.params('Experiment-Bucket')",
  "redirectFromDesktop": "$input.params('desktop-redirect')",
  "tt_medium": "$input.params('tt_medium')",
  "tt_content": "$input.params('tt_content')",
  "experiment_overrides": "$input.params('experiment_overrides')",
  "channel": "$input.params('channel')",
  "whatever": "$input.params('whatever')",
  "thirdThing": "$input.params('thirdThing')",
  "theRest": "$input.params('theRest')",
  "pathParts": ["$input.params('channel')", "$input.params('whatever')", "$input.params('thirdThing')", "$input.params('theRest')"]
}
END_OF_INTEGRATION_TEMPLATE
  }
}

resource "aws_api_gateway_deployment" "deployment" {
  depends_on = [
    "module.landing_page_endpoint",
    "module.games_directory_endpoint",
    "module.game_directory_endpoint",
    "module.channels_directory_endpoint",
    "module.video_viewer_endpoint",
    "module.channel_viewer_endpoint",
    "module.channel_chat_embed_endpoint",
    "module.channel_profile_endpoint",
    "module.channel_clips_endpoint",
    "module.channel_collections_endpoint",
    "module.channel_events_endpoint",
    "module.channel_followers_endpoint",
    "module.channel_following_endpoint",
    "module.channel_videos_endpoint",
    "module.channel_videos_category_endpoint",
    "module.channel_video_viewer_endpoint",
    "module.channel_upsell_level_2_endpoint",
    "module.channel_upsell_level_3_endpoint",
    "module.catchall_resource_endpoint",
    "module.event_details_endpoint",
  ]

  rest_api_id = "${aws_api_gateway_rest_api.mws_gateway.id}"
  stage_name  = "${var.api_stage}"

  variables = {
    "functionAlias" = "${var.api_stage}"
  }
}

output "url" {
  depends_on = ["aws_api_gateway_deployment.deployment"]
  value      = "${aws_api_gateway_deployment.deployment.invoke_url}/"
}
