variable "aws_region" {}

variable "api_stage" {}

variable integration_request_templates {
  type = "map"

  default = {
    "application/json" = "{\"statusCode\": 200}"
  }
}

variable response_templates {
  type = "map"

  # The meta refresh will load the app store.
  # However, once a user comes back to their browser we need something to still be on the page.
  # For this we'll load the mobile brochure site in an iframe since it seems relevant. It has
  # to be an iframe and not a redirect because a redirect will cause the modal to go to the app
  # store to fail in Safari (but not Chrome).
  default = {
    "application/json" = <<HTML_DOC
<html>
  <head>
    <meta http-equiv="refresh" content="0;URL='https://itunes.apple.com/app/apple-store/id460177396?pt=325952&ct=mwc-$util.escapeJavaScript($util.urlEncode($input.params('c')))&mt=8'" />
    <style>
      body {
        margin: 0;
        padding: 0;
        border: 0;
      }
    </style>
  </head>
  <body>
    <iframe src="https://mobile.twitch.tv" style="border: 0; width: 100%; height: 100%"></iframe>
  </body>
</html>
HTML_DOC
  }
}

variable integration_response_parameters {
  type = "map"

  default = {
    "method.response.header.Content-Type" = "'text/html; charset=utf-8'"
  }
}

variable request_parameters {
  type = "map"

  default = {
    # This parameter is part of the campaign string. For example, if the
    # campaign string was 'mwc-channel-titlebar' this would be 'channel-titlebar'.
    # It is injected in the response in the response template.
    "method.request.querystring.c" = true
  }
}

variable response_parameters {
  type = "map"

  default = {
    "method.response.header.Content-Type" = true
  }
}

###
#
# Gateway Configuration
#
###

resource "aws_api_gateway_rest_api" "ios_upsell_gateway" {
  description = "API Gateway for redirecting to iOS App Store page"
  name        = "iOS Upsell Gateway"
}

###
#
# Configuration for /
#
###

resource "aws_api_gateway_method" "root_method" {
  rest_api_id        = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id        = "${aws_api_gateway_rest_api.ios_upsell_gateway.root_resource_id}"
  http_method        = "ANY"
  authorization      = "NONE"
  request_parameters = "${var.request_parameters}"
}

resource "aws_api_gateway_integration" "root_method_integration" {
  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.root_resource_id}"
  http_method = "${aws_api_gateway_method.root_method.http_method}"

  type              = "MOCK"
  request_templates = "${var.integration_request_templates}"
}

# 200 response integration response
resource "aws_api_gateway_integration_response" "root_integration_response" {
  depends_on = [
    "aws_api_gateway_integration.root_method_integration",
  ]

  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.root_resource_id}"
  http_method = "${aws_api_gateway_method.root_method.http_method}"
  status_code = "${aws_api_gateway_method_response.root_success_response.status_code}"

  selection_pattern   = ""
  response_parameters = "${var.integration_response_parameters}"
  response_templates  = "${var.response_templates}"
}

# Method 200 Response
resource "aws_api_gateway_method_response" "root_success_response" {
  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.root_resource_id}"

  http_method = "${aws_api_gateway_method.root_method.http_method}"
  status_code = "200"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = "${var.response_parameters}"
}

###
#
# Configuration for /{all+}
#
###

resource "aws_api_gateway_resource" "proxy_resource" {
  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  parent_id   = "${aws_api_gateway_rest_api.ios_upsell_gateway.root_resource_id}"
  path_part   = "{all+}"
}

resource "aws_api_gateway_method" "proxy_method" {
  rest_api_id        = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id        = "${aws_api_gateway_resource.proxy_resource.id}"
  http_method        = "ANY"
  authorization      = "NONE"
  request_parameters = "${var.request_parameters}"
}

resource "aws_api_gateway_integration" "proxy_method_integration" {
  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id = "${aws_api_gateway_resource.proxy_resource.id}"
  http_method = "${aws_api_gateway_method.proxy_method.http_method}"

  type = "MOCK"

  request_templates = "${var.integration_request_templates}"
}

# 200 response integration response
resource "aws_api_gateway_integration_response" "proxy_integration_response" {
  depends_on = [
    "aws_api_gateway_integration.proxy_method_integration",
  ]

  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id = "${aws_api_gateway_resource.proxy_resource.id}"
  http_method = "${aws_api_gateway_method.proxy_method.http_method}"
  status_code = "${aws_api_gateway_method_response.proxy_success_response.status_code}"

  selection_pattern   = ""
  response_parameters = "${var.integration_response_parameters}"
  response_templates  = "${var.response_templates}"
}

# Method 200 Response
resource "aws_api_gateway_method_response" "proxy_success_response" {
  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  resource_id = "${aws_api_gateway_resource.proxy_resource.id}"

  http_method = "${aws_api_gateway_method.proxy_method.http_method}"
  status_code = "200"

  response_models = {
    "application/json" = "Empty"
  }

  response_parameters = "${var.response_parameters}"
}

###
#
# Deployment Configuration
#
###

resource "aws_api_gateway_deployment" "deployment" {
  depends_on = [
    "aws_api_gateway_integration_response.root_integration_response",
    "aws_api_gateway_method_response.root_success_response",
    "aws_api_gateway_integration_response.proxy_integration_response",
    "aws_api_gateway_method_response.proxy_success_response",
  ]

  rest_api_id = "${aws_api_gateway_rest_api.ios_upsell_gateway.id}"
  stage_name  = "${var.api_stage}"

  variables = {
    "functionAlias" = "${var.api_stage}"
  }
}
