# Name of the stage for the API, must match lambda alias
variable "api_stage" { }
# AWS account to spin things up, must be in lambda's account
variable "aws_account_id" { }
# Region
variable "aws_region" { }
#  function role for lambda, should ultimately come from apex.
variable "gateway_invoke_lambda_role_arn" { }
# Name of lambda, should ultimately come from apex.
variable "lambda_name" { }

provider "aws" {
  region = "${var.aws_region}"
}

resource "aws_iam_role" "allowAPIGatewayToAssumeRole" {
    name = "allowAPIGatewayToAssumeRole"
    assume_role_policy = <<END_OF_ROLE
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "apigateway.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
END_OF_ROLE
}

resource "aws_iam_role_policy" "polyfillGatewayExecution" {
  name = "polyfillGatewayExecution"
  role = "${aws_iam_role.allowAPIGatewayToAssumeRole.id}"
  policy = <<END_OF_POLICY
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "lambda:InvokeFunction"
            ],
            "Resource": "*"
        }
    ]
}
END_OF_POLICY
}

 resource "aws_api_gateway_rest_api" "polyfillGateway" {
   description = "API Gateway for Polyfill lambda"
   name = "Polyfill Gateway"
 }

# Resource /pf
resource "aws_api_gateway_resource" "pfEndpoint" {
  rest_api_id = "${aws_api_gateway_rest_api.polyfillGateway.id}"
  parent_id = "${aws_api_gateway_rest_api.polyfillGateway.root_resource_id}"
  path_part = "pf"
}

# Resource /pf/polyfill.min.js
resource "aws_api_gateway_resource" "minifiedPolyfill" {
  rest_api_id = "${aws_api_gateway_rest_api.polyfillGateway.id}"
  parent_id = "${aws_api_gateway_resource.pfEndpoint.id}"
  path_part = "polyfill.min.js"
}

# GET method for /pf/pollyfill.min.js
resource "aws_api_gateway_method" "getMinifiedPolyfill" {
  rest_api_id = "${aws_api_gateway_rest_api.polyfillGateway.id}"
  resource_id = "${aws_api_gateway_resource.minifiedPolyfill.id}"
  http_method = "GET"
  authorization = "NONE"
  request_parameters = {
    "method.request.header.User-Agent" = true,
    "method.request.querystring.excludes" = true,
    "method.request.querystring.features" = true,
    "method.request.querystring.flags" = true,
    "method.request.querystring.userAgent" = true,
  }
}

resource "aws_api_gateway_integration" "minifiedPolyfillGetIntegration" {
  rest_api_id             = "${aws_api_gateway_rest_api.polyfillGateway.id}"
  resource_id             = "${aws_api_gateway_resource.minifiedPolyfill.id}"
  http_method             = "${aws_api_gateway_method.getMinifiedPolyfill.http_method}"
  integration_http_method = "POST"
  type                    = "AWS"
  uri                     = "arn:aws:apigateway:${var.aws_region}:lambda:path/2015-03-31/functions/${var.lambda_name}/invocations"
  credentials             = "${aws_iam_role.allowAPIGatewayToAssumeRole.arn}"
  request_templates       = {
    "application/json" = <<JSON_REQUEST_TEMPLATE
{
  "excludes": "$input.params('excludes')",
  "features": "$input.params('features')",
  "flags": "$input.params('flags')",
  "queriedUserAgent": "$input.params('userAgent')",
  "minify": true,
  "userAgent": "$input.params('User-Agent')"
}
JSON_REQUEST_TEMPLATE
  }
}

# 200 response integration
resource "aws_api_gateway_integration_response" "polyfillGetResponseIntegration" {
  rest_api_id         = "${aws_api_gateway_rest_api.polyfillGateway.id}"
  resource_id         = "${aws_api_gateway_resource.minifiedPolyfill.id}"
  http_method         = "${aws_api_gateway_method.getMinifiedPolyfill.http_method}"
  status_code         = "${aws_api_gateway_method_response.getMinifiedPolyfillResponseSuccess.status_code}"
  # Success is specifically an empty error string.
  selection_pattern   = "^$"
  response_parameters = {
    "method.response.header.Vary" = "integration.response.body.vary"
  }
  response_templates = {
    "application/json" = "$input.path('$.content')"
  }
}

# Method 200 Response
resource "aws_api_gateway_method_response" "getMinifiedPolyfillResponseSuccess" {
 rest_api_id = "${aws_api_gateway_rest_api.polyfillGateway.id}"
 resource_id = "${aws_api_gateway_resource.minifiedPolyfill.id}"
 
http_method = "${aws_api_gateway_method.getMinifiedPolyfill.http_method}"
  status_code = "200"
  response_models = {
    "application/json" = "Empty"
  }
  response_parameters = {
    "method.response.header.Vary" = true
  }
}

# 500 response integration
resource "aws_api_gateway_integration_response" "polyfillGetResponseIntegrationForError" {
  rest_api_id         = "${aws_api_gateway_rest_api.polyfillGateway.id}"
  resource_id         = "${aws_api_gateway_resource.minifiedPolyfill.id}"
  http_method         = "${aws_api_gateway_method.getMinifiedPolyfill.http_method}"
  status_code         = "${aws_api_gateway_method_response.getMinifiedPolyfillResponseInternalError.status_code}"
  selection_pattern   = "-"
}

# Method 500 Response
resource "aws_api_gateway_method_response" "getMinifiedPolyfillResponseInternalError" {
 rest_api_id = "${aws_api_gateway_rest_api.polyfillGateway.id}"
 resource_id = "${aws_api_gateway_resource.minifiedPolyfill.id}"
 
  http_method = "${aws_api_gateway_method.getMinifiedPolyfill.http_method}"
  status_code = "500"
  response_models = {
    "application/json" = "Empty"
  }
}

resource "aws_api_gateway_deployment" "deployment" {
  depends_on = [
    "aws_api_gateway_integration.minifiedPolyfillGetIntegration",
    "aws_api_gateway_integration_response.polyfillGetResponseIntegration"
  ]
  rest_api_id = "${aws_api_gateway_rest_api.polyfillGateway.id}"
  stage_name  = "${var.api_stage}"
  variables   = {
    "functionAlias" = "${var.api_stage}"
  }
}

