Spud
======

## spud: noun
1. A potato 

2. A small, narrow spade for cutting the roots of plants, especially weeds.

3. A short length of pipe that is used to connect two components or that takes the form of a projection from a fitting to which a pipe may be screwed.

Spud is a configurable TypeScript Lambda Template developed by Systems Infrastructure that supports moving data from a compressed [Spade Kinesis Stream](https://git-aws.internal.justin.tv/twitch/docs/tree/master/datainfra/kinesis) into CloudWatch as Custom Metrics.

Metrics are combined into minutely distributions with your defined Dimensions (and rollup dimensions), compressed with SEH1 and uploaded to CloudWatch in a very similar way to the in-service Go Library [TwitchTelemetry](https://git-aws.internal.justin.tv/amzn/TwitchTelemetry).
```
                                     +--------+
                                     |        |
                                +--->+  Spud  +----+
                                |    |        |    |
                                |    +--------+    |
                                |                  |         +------------------------+
     +-----------------+        |    +--------+    |         |                        |
     |                 |        |    |        |    |         |       CloudWatch       |
     |  Spade Kinesis  +------------>+  Spud  +-------------->                        |
     |    (3 shards)   |        |    |        |    |         |     PutMetricData()    |
     +-----------------+        |    +--------+    |         |  (SEH1 Distributions)  |
                                |                  |         +------------------------+
                                |    +--------+    |
                                |    |        |    |
                                +--->+  Spud  +----+
                                     |        |
                                     +--------+
```

## Quick Start

* Create a blank `spud-<yourproject>` repository in your team's GHE Org
* Clone Spud into a new folder and replace the origin with your new repo
```bash
git clone git@git-aws.internal.justin.tv:systems/spud.git spud-<yourproject>
cd spud-<yourproject>
git remote rename origin upstream
git remote add origin git@git-aws.internal.justin.tv:<ORG>/spud-<yourproject>.git
git push -u origin master
```
* Install [`nvm`](https://github.com/nvm-sh/nvm/blob/master/README.md#installing-and-updating) or [`nodenv`](https://github.com/nodenv/nodenv#installation)
* Navigate to `spud-<yourproject>` and install node, npm, and your dependencies
```bash
cd spud-<yourproject>
nodenv install # Installs NodeJS and NPM according to the version listed in .node-version
npm install # Installs the dependencies listed in package.json
```
* Copy `./config/spud.json.dist` over `./config/spud.json` and update with your Service Name (<yourproject>), compressed Spade Kinesis ARN(s), Credentials profile name(s)
  * Change the namespace from `Twitch` to your preferred CloudWatch namespace. This will appear in your CloudWatch Metrics dashboard.
* Run the refresh-data script to generate a fake lambda event with real data from your prod stream
```
npm run refresh-data
# or `./node_modules/.bin/ts-node refresh_test_data.ts dev > test_data.json` for dev stream data
```
* Run the `dev:raw` command to output the raw events from your kinesis stream
* Run the `dev` command to output the processed data points that will be written to CloudWatch
  * If you do not see CloudWatch output, check your filters to ensure they are correctly parsing the Kinesis stream.
```
npm run dev:raw
npm run dev
```
* Configure your events/metrics according to the configuration methods below
  * For basic use cases, create a TOML filter in `config/spade_events/` for each handled event from your Spud config
* Deploy to AWS via Serverless/CloudFormation
```
npm run deploy # Dev Account
npm run deploy:prod # Prod Account
```
  
----

## Update Spud

To update your instance of Spud to the latest version in the main Spud repository, first ensure you have a git remote added for `spud.git`
```bash
git remote add upstream git@git-aws.internal.justin.tv:systems/spud.git
```
Then checkout the files in `src/` from that remote
```bash
git fetch upstream master
git checkout upstream/master -- src refresh_test_data.ts handler.ts package.json package-lock.json tsconfig.json webpack.config.js
```

## TOML Metric Configuration

The `npm run dev:raw` command shows that one of the metrics we're interested in looks as follows:
```json
{
    "Name": "benchmark_complete_transition",
    "Fields": {
        "app_version": "19aef467-f884-4a71-9834-c82df7bfa057",
        "city": "South San Francisco",
        "client_app": "twilight",
        "country": "US",
        "destination": "channel.index.index",
        "domain": "twitch.tv",
        "is_app_launch": "false",
        "logged_in": "true",
        "lost_visibility": "true",
        "page_component_name": "ChannelPage",
        "page_session_id": "dffafcb269200c21",
        "platform": "web",
        "region": "CA",
        "time_from_fetch": "2346",
        "time_utc": "2019-08-30 06:31:20",
        "url": "https://www.twitch.tv/dogdog"
    }
}
```

You can create a configuration for this metric purely in TOML (similar to Plucker) using the same filename as the `Name` field above. `./config/spade_events/benchmark_complete_transition.toml` 

```toml
# Minimum TOML
[Transition] # What the Metric name in CloudWatch will be, required
    # The field from the raw data where the `value` for this metric comes from, will be converted to a JS `number`
    metric_field = "time_from_fetch" # required

    # Must exactly match a Unit name from https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html
    unit = "Milliseconds" # required
```

For basic Count metrics, use the synthetic metric field `spud_event_counter`.

If you need "rollups" of CloudWatch data (i.e. viewing data without segmenting by dimensions), see the `dimension_groups` and `rollup` features in the full TOML reference below.

Add the event name into `./config/spud.json` under `handled_events` to load this configuration file:
```json
"events": {
    "namespace": "Twilight",
    "handled_events": [
        "benchmark_complete_transition"
    ]
}
```

When you run `npm run dev` with the above configuration, you should see a CloudWatch MetricDatum Array printed to your console:
```json
{
  "MetricData": [
    {
      "StatisticValues": {
        "SampleCount": 271,
        "Sum": 3584429,
        "Minimum": 6,
        "Maximum": 1920553
      },
      "MetricName": "Transition",
      "Timestamp": "2019-08-30T06:29:00.000Z",
      "Unit": "Milliseconds",
      "Dimensions": [],
      "Values": [
        1614.1098945536328,
        1775.5208840089958,
        2363.218296615975,
        "..."
      ],
      "Counts": [
        22,
        14,
        17,
        "..."
      ]
    },
```

The above MetricDatum are compressed with [Amazon's SEH1 algorithm](https://w.amazon.com/index.php/MonitoringTeam/MetricAgent/SEHAggregation) to attempt to save on request costs (*very* similar to how [TwitchTelemetry](https://git-aws.internal.justin.tv/amzn/TwitchTelemetry) handles this).

```toml
# Full Reference
handler = false # Don't try to load a custom TypeScript class for parsing this event (the default)

[PageLoad] # What the Metric name in CloudWatch will be
    # Override the Metric name from above
    metric_name = "PageLoad" # Defaults to the TOML Table Name ('[PageLoad]')

    # The field from the raw data where the `value` for this metric comes from, will be converted to a JS `number`
    metric_field = "time_from_fetch"

    # The CloudWatch Unit for the Value Above
    # Must exactly match a Unit name from https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html
    unit = "Milliseconds"

    # Pick Field Names from the raw data to use as CloudWatch Dimensions
    # More Cardinality = More cost for these uploads, try to avoid anything highly variable
    # Cloudwatch allows a maximum of 10 Dimensions per Metric
    dimensions = [
        "country",
        "region",
        "city",
        "logged_in"
    ]

    # Create additional metrics excluding each combination of rollup dimensions
    rollup_dimensions = [
        ["region", "city"], # Create a new metric with Country/Logged In as the only dimensions
        ["region", "city", "logged_in"] # Create a new metric with Country as the only dimension
    ]
    
    # This is an alternative to dimensions and rollup_diemnsions you cannot use them both at the same time. The below
    # configuration would produce the same set of metrics as the above dimensions and rollup_dimensions configuration.
    # This is here so users have the flexiblity to use whichever configuration works best for them.
    dimension_groups = [
        ["country"],
        ["country", "logged_in"],
        ["country", "region", "city", "logged_in"]
    ]



    # Rename a field
    [PageLoad.alias]
    app = "client_app" # Create or overwrite the field 'app' using the value of the 'client_app' field

    # Manually set a field
    [PageLoad.synthetic]
    client_app = "any" # Create or overwrite the field 'client_app' with the static value 'any'

    # Filters are ANDed together and select the events that will be included in this Metric
    [PageLoad.filter.equal]
    # The lost_visibility field in the Spade Event must equal `false`
    lost_visibility = "false"
    # The client_app field must equal `twilight`
    client_app = "twilight"
    # The city field must equal `Seattle` or `San Francisco`
    city = ["Seattle", "San Francisco"]

    [PageLoad.filter.not_equal]
    # The is_app_launch field must not equal `false`
    is_app_launch = "false"
    # The region field must not equal `mars` or `venus`
    region = ["mars", "venus"]

    [PageLoad.filter.regex]
    # The domain field must end with `twitch.tv`
    domain = "twitch\\.tv$"
    # The path field must end with `twitch.tv` or start with `localhost.twitch`
    path = ["twitch\\.tv$", "^localhost\\.twitch"]

    [PageLoad.filter.not_regex]
    country = "(Anonymous Proxy|Satellite Provider)"
    city = ["^[A-Z]", "^[^*]+$"]

    [PageLoad.filter.value_range]
    # 0 <= time_from_fetch <= 120000 milliseconds
    time_from_fetch = "0..120000"
    # 0 <= time_from_create <= 12 OR 18 <= time_from_create <= 36
    time_from_create = ["0..12", "18..36"]
```
----

## Custom TypeScript Parser
 
Given the Spade raw data (from `npm run dev:raw`):
```json
  {
    "Name": "network_request",
    "Fields": {
      "app_version": "19aef467-f884-4a71-9834-c82df7bfa057",
      "city": "Philadelphia",
      "client_app": "twilight",
      "connect_duration": "0",
      "country": "US",
      "destination": "channel.index.index",
      "dns_duration": "0",
      "domain": "twitch.tv",
      "duration": "3",
      "is_pre_pageload": "false",
      "logged_in": "true",
      "page_component_name": "ChannelPage",
      "page_session_id": "895975a6f9bc430f",
      "platform": "web",
      "redirect_duration": "",
      "region": "PA",
      "relative_start_time": "985423",
      "request_duration": "0",
      "request_url": "https://static-cdn.jtvnw.net/emoticons/v1/425618/1.0",
      "response_duration": "0",
      "start_time_ts": "",
      "start_time_utc_ts": "",
      "time_utc": "2019-08-30 06:31:21",
      "transfer_size": "",
      "type": "resource",
      "url": "https://www.twitch.tv/felixiima"
    }
  }
```

You can create a configuration for a metric with a combination of TOML (above) and your own TypeScript class using the same filename as the `Name` field in the Spade data.

`./config/spade_events/network_request.toml`:
```toml
handler = true # Use DataPoint implementation from ./config/spade_events/models/network_request.ts for this

[NetworkTransferSize]
    metric_field = "transfer_size"
    unit = "Bytes"

    dimensions = [
        "country"
    ]

    [NetworkTransferSize.filter.not_equal]
    browser_cache_hit = "true" # Synthetic Value not in Spade data, created below
```

`./config/spade_events/models/network_request.ts`:
```typescript
import { SpadeEventFields } from "../../../src/spade_event_data";
import { SpudConfigData } from "../../../src/config";
import SpadeDataPoint from "../../../src/spade_data_point";

// This must be a `default` export and implement DataPoint (or extend SpadeDataPoint)
export default class NetworkRequest extends SpadeDataPoint {

  constructor(readonly event_config: SpudConfigData, fields: SpadeEventFields) {
    super(event_config, fields); // Run the standard `SpadeDataPoint` constructor
    this.mungeData(); // Run our own function modifying the Spade Event Data
  }

  private mungeData() {
    // Rename `start_time_utc_ts` to `start_time` (pre-filter/dimension checks from TOML)
    this.fields.start_time = +this.fields.start_time_utc_ts;
    delete this.fields.start_time_utc_ts;

    // Create a new value as the sum of two others
    //  this new value can be referenced in the TOML config
    //  (filter/dimension/metric_field)
    if (this.fields.hasOwnProperty('relative_start_time')) {
      this.fields.relative_complete_time = +this.fields.duration + +this.fields.relative_start_time;
    } else {
      this.fields.relative_start_time = 0;
      this.fields.relative_complete_time = 0;
    }

    // Create a new boolean, can be used in filters/etc
    this.fields.browser_cache_hit = this.fields.transfer_size == 0 ? 'true' : 'false'
  }

}
```
