import { regex } from "./regex";
import { SpadeEventFields } from "../../../src/spade_event_data";
import { SpudConfigData } from "../../../src/config";
import SpadeDataPoint from "../../../src/spade_data_point";
import { Logger } from "../../../src/log-aggregator";

export default class NetworkRequest extends SpadeDataPoint {

  constructor(readonly event_config: SpudConfigData, fields: SpadeEventFields, logger: Logger) {
    super(event_config, fields, logger);
    this.mungeData();
  }

  private mungeData() {
    this.fields.start_time = +this.fields.start_time_utc_ts;
    delete this.fields.start_time_utc_ts;

    if (this.fields.hasOwnProperty('relative_start_time')) {
      this.fields.relative_complete_time = +this.fields.duration + +this.fields.relative_start_time;
    } else {
      this.fields.relative_start_time = 0;
      this.fields.relative_complete_time = 0;
    }

    this.fields.cdn = this.cdn();
    this.fields.request_type = this.request_type();

    this.fields.browser_cache_hit = this.fields.transfer_size == 0 ? 'true' : 'false'
  }

  private request_type(): string {
    switch (true) {
      case regex.IsCoreJS.test(this.fields.request_url as string):
        return 'Static - Core.js';
      case regex.IsSettingsJS.test(this.fields.request_url as string):
        return 'Static - Settings.js';
      case regex.IsManifestJS.test(this.fields.request_url as string):
        return 'Static - Manifest.js';
      case regex.IsVendorJS.test(this.fields.request_url as string):
        return 'Static - Vendor.js';
      case regex.IsCoreCSS.test(this.fields.request_url as string):
        return 'Static - Core.css';
      case regex.IsWorkerJS.test(this.fields.request_url as string):
        return 'Static - Worker.js';
      case regex.IsPolyfillJS.test(this.fields.request_url as string):
        return 'Static - Polyfill.js';
      case regex.IsTwilightImage.test(this.fields.request_url as string):
        let image_matches = regex.IsTwilightImage.exec(this.fields.request_url as string);
        if (image_matches) {
          return `Static - Images - ${image_matches[1]}`;
        } else {
          return 'Static - Images';
        }
      case regex.IsPageJS.test(this.fields.request_url as string):
        let page_matches = regex.IsPageJS.exec(this.fields.request_url as string);
        if (page_matches) {
          return `Static - Pages.js - ${page_matches[1]}`;
        } else {
          return 'Static - Pages.js';
        }
      case regex.IsLocalesJS.test(this.fields.request_url as string):
        let locale_matches = regex.IsLocalesJS.exec(this.fields.request_url as string);
        if (locale_matches) {
          return `Static - Locale.js - ${locale_matches[1]}`;
        } else {
          return 'Static - Locale.js';
        }
      case regex.IsEntryJS.test(this.fields.request_url as string):
        let entry_matches = regex.IsLocalesJS.exec(this.fields.request_url as string);
        if (entry_matches) {
          return `Static - Entry.${entry_matches[2]} - ${entry_matches[1]}`;
        } else {
          return 'Static - Entry';
        }
      case regex.IsFavicon.test(this.fields.request_url as string):
        return 'Static - Favicon';
      case regex.IsCustos.test(this.fields.request_url as string):
        return 'Static - SourcePoint';
      case regex.IsGraphQL.test(this.fields.request_url as string):
        return 'GraphQL';
      case regex.IsKraken.test(this.fields.request_url as string):
        let kraken_matches = regex.IsKraken.exec(this.fields.request_url as string);
        if (kraken_matches) {
          return `API - Kraken - ${kraken_matches[3]}`;
        } else {
          return 'API - Kraken';
        }
      case regex.IsViewerToken.test(this.fields.request_url as string):
        return 'API - Viewer Token';
      case regex.IsAPIMe.test(this.fields.request_url as string):
        return 'API - Me'
      case regex.IsProductionSage.test(this.fields.request_url as string):
        return 'Sage';
      case regex.IsJtvnw.test(this.fields.request_url as string):
        let jtvnw_matches = regex.IsJtvnw.exec(this.fields.request_url as string);
        if (jtvnw_matches) {
          return `JTVNW - ${jtvnw_matches[2]}`;
        } else {
          return 'JTVNW - Other';
        }
      case regex.IsPlayerJS.test(this.fields.request_url as string):
        return 'Player - player.js';
      case regex.IsPlayerCSS.test(this.fields.request_url as string):
        return 'Player - player.css';
      case regex.IsPlayerOther.test(this.fields.request_url as string):
        return 'Player - Other'
      case regex.IsClientReporter.test(this.fields.request_url as string):
        return 'Client Reporter';
      case regex.IsClipsMediaAssets.test(this.fields.request_url as string):
        return 'Clips - Media Assets';
      case regex.IsVodStoryboards.test(this.fields.request_url as string):
        return 'VOD Storyboards';
      case regex.IsSentinel.test(this.fields.request_url as string):
        return 'Sentinel';
      case regex.IsPubSub.test(this.fields.request_url as string):
        return 'PubSub - Websocket';
      case regex.IsPubster.test(this.fields.request_url as string):
        return 'PubSub - Pubster';
      case regex.IsIRC.test(this.fields.request_url as string):
        return 'Chat - Websocket'
      case regex.IsClips.test(this.fields.request_url as string):
        return 'Clips'
      default:
        return 'Uncategorized';
    }
  }

  cdn(): string {
    switch (true) {
      case regex.IsAkamai.test(this.fields.request_url as string):
        return 'Akamai';
      case regex.IsCloudFront.test(this.fields.request_url as string):
        return 'CloudFront';
      case regex.IsCloudFlare.test(this.fields.request_url as string):
        return 'CloudFlare';
      case regex.IsFastly.test(this.fields.request_url as string):
        return 'Fastly';
      case regex.IsControl.test(this.fields.request_url as string):
        return 'Control'
      default:
        return ''
    }
  }
}
