import { Distribution } from "./distribution";
import { Sample } from "./sample";

// keySeparator is used to separate values when converting a metric id into a unique key for use in maps.
// Metric names and dimension names should not contain newlines.
const KEY_SEPARATOR = "\n";

// Aggregator keeps a distribution for each unique metric ingested
export class Aggregator {
  aggregationPeriod: number;
  distributions: { [name: string]: Distribution } = {};

  // Set aggregationPeriod to the time resolution needed, for example time.Minute for one-minute metrics.
  // Set aggregationPeriod to zero to disable this time bucketing and aggregate samples regardless of timestamp.
  public constructor(aggregationPeriod: number) {
    this.aggregationPeriod = aggregationPeriod;
  }

  // aggregateSample takes a sample and aggregates it into
  // the distribution for the sample's metric. If the distribution
  // does not exist, it will be created.
  public aggregateSample(sample: Sample) {
    let aggregationTimestamp = this.alignToAggregationPeriod(this.aggregationPeriod, sample.Timestamp);
    let hashKey = this.uniqueKeyForMap(sample, aggregationTimestamp, this.aggregationPeriod);

    let distribution = this.distributions[hashKey];

    if (!distribution) {
      distribution = new Distribution(sample);
      distribution.Timestamp = aggregationTimestamp;
      this.distributions[hashKey] = distribution;
    } else {
      distribution.addSample(sample);
    }
  }

  // getDistributions returns all distributions in the Aggregator.
  // Does not modify or clear the distributions
  // Note that if the aggregator is not reset() after a call to getDistributions()
  // then the distributions can continue to be modified by called to AggregateSample()
  private getDistributions(): Distribution[] {
    let distributions: Distribution[] = [];
    for (let d of Object.values(this.distributions)) {
      distributions.push(d);
    }
    return distributions;
  }

  // Reset wipes the Aggregator clean by deleting all Distributions
  public reset() {
    this.distributions = {};
  }

  // Flush returns current distributions, while also resetting them internally
  public flush(): Distribution[] {
    let distributions = this.getDistributions();
    this.reset();
    return distributions;
  }

  private alignToAggregationPeriod(aggregationPeriod: number, timestamp: Date): Date {
    if (aggregationPeriod > 0) {
      const epochMs = timestamp.getTime()
      const offset = epochMs % aggregationPeriod;
      return new Date(epochMs - offset)
    }
    return timestamp
  }

  private uniqueKeyForMap(sample: Sample, aggregationTimestamp: Date, aggregationPeriod: number): string {
    let s = sample.MetricID.Name

    if (aggregationPeriod > 0) {
      const timeKey = Math.floor(aggregationTimestamp.getTime() / 1000)
      s = s.concat(KEY_SEPARATOR, timeKey.toString())
    }

    for (let k of Object.keys(sample.MetricID.Dimensions).sort()) {
      s = s.concat(KEY_SEPARATOR, k, KEY_SEPARATOR, sample.MetricID.Dimensions[k])
    }
    return s.slice(0, 500)
  }

}
