import AWS = require('aws-sdk');
import { Distribution } from "./distribution";
import { StatisticSet, MetricDatum, Dimensions } from 'aws-sdk/clients/cloudwatch';
import { DimensionSet } from './metric_id';

const CW_MAX_DATUMS = 100;
const MAX_VALUES_PER_DATUM = 5000;

export class SpudCloudWatch {
  static client = new AWS.CloudWatch()

  public static async send(namespace: string, cwCollection: MetricDatum[]) {
    if (cwCollection.length == 0) { return }

    let promises: Promise<any>[] = [];

    // We want to split metrics every CW_MAX_DATUMS elements.
    for (let i = 0; i < cwCollection.length; i += CW_MAX_DATUMS) {
      // Determine the next endpoint
      let colEnd = i + CW_MAX_DATUMS;
      if (colEnd > cwCollection.length) { colEnd = cwCollection.length }

      // Send the sub-collection
      let cwSlice = cwCollection.slice(i, colEnd);
      promises.push(this.sendMetrics(namespace, cwSlice).catch(err => {
        console.log(err);
        if (err.code && err.code == "RequestEntityTooLarge") {
          return this.halveLargeRequest(namespace, cwSlice);
        }
      }));
    }

    return Promise.all(promises);
  }

  // halveLargeRequest allows us to break up requests that were found to be too large by halving the collection size and
  // trying again, calling breakupLargeRequests to further break down the request until it is either small enough or a
  // single datum remains
  private static async halveLargeRequest(namespace: string, cwCollection: MetricDatum[]): Promise<any> {
    // If our length is 1, we don't split further. Technically, it is possible for this to be greater than the
    // request limit (since we are sending two arrays of float64 of max size 5000), but this is incredibly unlikely
    // since all values are likely to have been compressed by this point.
    if (cwCollection.length == 1) {
      console.log("Failed to publish a single metric (too large?): " + JSON.stringify(cwCollection, null, 2));
      return Promise.resolve();
    }

    let halfStep = Math.ceil(cwCollection.length / 2.0)
    let firstHalf = cwCollection.slice(0, halfStep);
    let secondHalf = cwCollection.slice(halfStep, cwCollection.length);

    let firstReq = this.sendMetrics(namespace, firstHalf).catch(err => {
      if (err.code && err.code == "InvalidParameterValue" || err.code == "RequestEntityTooLarge") {
        return this.halveLargeRequest(namespace, firstHalf);
      }
      console.log(err)
    });

    let secondReq = this.sendMetrics(namespace, secondHalf).catch(err => {
      if (err.code && err.code == "InvalidParameterValue" || err.code == "RequestEntityTooLarge") {
        return this.halveLargeRequest(namespace, secondHalf);
      }
      console.log(err)
    });

    return Promise.all([firstReq, secondReq])
  }

  // sendMetrics performs the CloudWatch PutMetricData API call
  private static async sendMetrics(namespace: string, cwCollection: MetricDatum[]) {
    if (cwCollection.length == 0) { return Promise.resolve(); }
    let params = { MetricData: cwCollection, Namespace: namespace }

    // Just print out the PutMetricData params when run locally (sls invoke --local)
    if (process.env.IS_LOCAL) {
      if (process.env.PRINT_CLOUDWATCH) {
        console.log(JSON.stringify(params, null, 2));
      }
      return Promise.resolve()
    }

    return this.client.putMetricData({ MetricData: cwCollection, Namespace: namespace }).promise()
  }

  public static constructMetricDatums(distributions: Distribution[]): MetricDatum[] {
    let allDatum: MetricDatum[] = [];
    for (let element of distributions) {
      let newDatums = this.constructNewDatums(element);
      if (newDatums.length == 0) { continue; }
      allDatum.push(...newDatums)
    }
    return allDatum;
  }

  private static constructNewDatums(distribution: Distribution): MetricDatum[] {
    if (distribution.MetricID.Name == "") { return [] }

    // A single telemetry distribution has all related dimensions. These need to be broken into separate CW
    // datums to allow for rollups (since CW does not do this automatically).
    // This means n dimensions in the dimension list for one internal metric = n separate CW datums
    let cwDatums: MetricDatum[] = [];

    // Create the single CW datum that will serve as the base for every other datum
    let baseDatum: MetricDatum;

    // CW now supports values and counts. Use the histogram to construct these if possible
    // Each datum can have a maximum of maxValuesPerDatum unique values. We therefore need to make sure we break up the
    // values as they come along and construct separate datums if necessary
    let allValues: number[][] = [];
    let allCounts: number[][] = [];

    if (distribution.SEH1) {
      let allValuesIdx = 0;
      allValues.push([])
      allCounts.push([])

      // Set the zero bucket if required
      if (distribution.SEH1.ZeroBucket != 0) {
        allValues[allValuesIdx].push(0.0)
        allCounts[allValuesIdx].push(distribution.SEH1.ZeroBucket)
      }

      // Set the individual values
      for (let [bucketVal, sampleCount] of distribution.SEH1.Histogram.entries()) {
        // As noted, there cannot be more than maxValuesPerDatum unique values per datum
        if (allValues[allValuesIdx].length >= MAX_VALUES_PER_DATUM) {
          allValuesIdx++;
          allValues.push([])
          allCounts.push([])
        }

        let approximateVal = distribution.SEH1.approximateOriginalValue(bucketVal);
        allValues[allValuesIdx].push(approximateVal)
        allCounts[allValuesIdx].push(sampleCount)
      }
    }

    let statisticalSet: StatisticSet = {
      SampleCount: distribution.SampleCount,
      Sum: distribution.Sum,
      Minimum: distribution.Minimum,
      Maximum: distribution.Maximum
    }

    baseDatum = {
      StatisticValues: statisticalSet,
      MetricName: distribution.MetricID.Name,
      Timestamp: distribution.Timestamp,
      Unit: distribution.Unit
    }

    // If the length of allValues > 0, we had a usable SEH1, so we need to construct 1 or more defaultMetric depending
    // on the number of unique values our histogram had
    if (allValues.length > 0) {
      for (let [allValIdx, valueList] of allValues.entries()) {
        cwDatums.push(...this.createRollupDatums(baseDatum, distribution.MetricID.Dimensions, distribution.RollupDimensions, valueList, allCounts[allValIdx]))
      }
    } else {
      // Use the baseDatum to construct all new datums, but don't set any values or counts
      cwDatums.push(...this.createRollupDatums(baseDatum, distribution.MetricID.Dimensions, distribution.RollupDimensions))
    }

    return cwDatums;
  }

  private static createRollupDatums(baseDatum: AWS.CloudWatch.MetricDatum, allDims: DimensionSet, distRollupDims: string[][], valueList: number[] = [], countsList: number[] = []): MetricDatum[] {
    let datumList: MetricDatum[] = [];

    // Start by determining the base dimensions (these will be used for a single datum)
    let defaulDims = this.constructDimensions(allDims);

    // Use the baseDatum to construct all new datums
    let defaultMetric = baseDatum;
    defaultMetric.Dimensions = defaulDims;

    if (valueList.length > 0) { defaultMetric.Values = valueList }
    if (countsList.length > 0) { defaultMetric.Counts = countsList }

    datumList.push(defaultMetric);

    // Construct the appropriate roll ups
    for (let rollupDims of distRollupDims) {
      if (rollupDims === undefined || rollupDims.length == 0) { continue; }
      let rollupDim = this.createRollupDimensions(allDims, rollupDims);
      let rollupMetric: MetricDatum = Object.assign({}, baseDatum);
      rollupMetric.Dimensions = rollupDim;

      if (valueList.length > 0) { rollupMetric.Values = valueList }
      if (countsList.length > 0) { rollupMetric.Counts = countsList }

      datumList.push(rollupMetric);
    }

    return datumList;
  }

  private static createRollupDimensions(allDims: DimensionSet, rollups: string[]): Dimensions {
    // Copy the original map
    let dims: DimensionSet = {};
    for (let [name, value] of Object.entries(allDims)) {
      dims[name] = value;
    }

    // Remove all the rollup dimensions from the original dimension map
    for (let name of rollups) {
      delete dims[name];
    }

    // Convert the updated dimension map into CW dimensions
    return this.constructDimensions(dims);
  }

  private static constructDimensions(dimensions: DimensionSet): Dimensions {
    let cwDims: Dimensions = [];

    for (let [name, value] of Object.entries(dimensions)) {
      if (name == "" || value == "") { continue; }

      cwDims.push({
        Name: name,
        Value: value
      });
    }

    return cwDims;
  }

}
