import SpadeDataPoint from "./spade_data_point";

export type FilterValue = string | string[]

export interface FilterConfig {
  equal?: { [field: string]: FilterValue };
  not_equal?: { [field: string]: FilterValue };
  regex?: { [field: string]: FilterValue };
  not_regex?: { [field: string]: FilterValue };
  value_range?: { [field: string]: FilterValue };
}

export interface MetricConfigData {
  metric_name: string;
  metric_field: string;
  dimensions?: string[];
  dimension_groups?: string[][];
  rollup_dimensions?: string[][];
  filter: FilterConfig;
  synthetic?: { [key: string]: string };
  alias?: { [key: string]: string };
  [name: string]: any;
}

type EventHandler = {
  [metricName: string]: MetricConfigData;
} & {
  handler?: typeof SpadeDataPoint;
}

interface EventHandlers {
  [eventName: string]: EventHandler
}

export type MetricData = Omit<MetricConfigData, 'dimension_groups'>

export interface SpudConfigData {
  event_name: string;
  handler?: typeof SpadeDataPoint;
  metrics: MetricData[];
}

export interface SpudConfigOpts {
  namespace: string;
  eventHandlers: EventHandlers;
}

export class SpudConfig {
  namespace: string;
  preparedEvents: { [name: string]: SpudConfigData } = {};

  constructor({ namespace, eventHandlers }: SpudConfigOpts) {
    this.namespace = namespace;

    for (let [event_name, eventHandler] of Object.entries(eventHandlers)) {
      let eventConfig: SpudConfigData = {
        event_name,
        handler: eventHandler.handler,
        metrics: []
      }

      for (let [metricName, metricConfig] of Object.entries(eventHandler)) {
        if (metricConfig.metric_field) {

          if (metricConfig.dimensions && metricConfig.dimension_groups) {
            throw ("Specifying dimensions and dimension_groups is not supported for " + event_name + "/" + metricName)
          }

          if (metricConfig.dimension_groups) {
            const flattenedDimensions = metricConfig.dimension_groups.reduce(
              (acc, group) => acc.concat(group),
              []
            )
            const uniqueDimensions = Array.from(new Set(flattenedDimensions))
            metricConfig.dimensions = uniqueDimensions

            metricConfig.rollup_dimensions = metricConfig.dimension_groups.map((group) => {
              return uniqueDimensions.filter((dim) => !group.includes(dim))
            })
            .filter((rollup) => rollup.length > 0)

            delete metricConfig.dimension_groups
          }

          if (metricConfig.dimensions && metricConfig.dimensions.length > 10) {
            throw ("More than 10 dimensions specified for " + event_name + "/" + metricName)
          }

          metricConfig.metric_name = metricConfig.metric_name || metricName
          eventConfig.metrics.push(metricConfig)
        }
      }

      this.preparedEvents[event_name] = eventConfig;
    }
  }

  public getEventConfig(event_name: string): SpudConfigData | undefined {
    return this.preparedEvents[event_name]
  }
}
