import { SpadeEventFields } from "./spade_event_data";
import { Logger } from "./log-aggregator";
import { FilterConfig, FilterValue } from "./config";

export class filters {

  public static testAll(fields: SpadeEventFields, filters: FilterConfig = {}, logger: Logger): boolean {
    for (let [filter_name, f] of Object.entries(filters)) {
      if (!f) { continue; }
      for (let [field_name, filter] of Object.entries(f)) {
        if (typeof filter != 'string' && !Array.isArray(filter)) {
          logger.log(`Filter ${field_name} ${filter_name} '${filter}' isn't a string or array, skipping`)
          continue;
        }
        if (!this.test(filter_name, filter, fields[field_name], logger)) {
          return false
        }
      }
    }
    return true
  }

  public static test(filter_name: string, filter: FilterValue, test_data: string | number, logger: Logger): boolean {
      if (Array.isArray(filter)) {
        return filter.some(f => {
          if (typeof f != 'string') {
            logger.log(`Filter values for ${filter_name} contain a non-string value: ${f}`)
            return false
          }
          return this.testSingle(filter_name, f, test_data, logger)
        })
      } else {
        return this.testSingle(filter_name, filter, test_data, logger)
      }
  }

  private static testSingle(filter_name: string, filter: string, test_data: string | number, logger: Logger): boolean {
    switch (filter_name) {
      case 'equal':
        return this.equal(filter, test_data);
      case 'not_equal':
        return !this.equal(filter, test_data);
      case 'regex':
        return this.regex(filter, test_data);
      case 'not_regex':
        return !this.regex(filter, test_data);
      case 'value_range':
        return this.range(filter, test_data, logger);
      default:
        logger.log(`Filter '${filter_name}' not implemented, failing open (everything matches)`)
        return true
    }
  }

  private static equal(filter: string, test_data: string | number): boolean {
    return filter == test_data
  }

  private static regex(filter: string, test_data: string | number): boolean {
    let regex = RegExp(filter);
    return regex.test(test_data.toString())
  }

  private static range(filter: string, test_data: string | number, logger: Logger): boolean {
    if (filter.match(/^(\d+)\.\.(\d+)$/)) {
      let min = parseFloat(RegExp.$1)
      let max = parseFloat(RegExp.$2)
      if (min < max) {
        return (test_data >= min && test_data <= max)
      } else {
        logger.log(min + " is not less than " + max + " in range filter " + filter + " failing open")
        return true
      }
    } else {
      logger.log("range filter '" + filter + "' does not match '\d+..\d+' format, failing open")
      return true
    }
  }

}
