import now = require('performance-now');
import { Aggregator } from './aggregator';
import { LogAggregator } from './log-aggregator';
import { Sample } from './sample';
import { SpadeEvent } from './spade_event'
import { SpudCloudWatch } from './cloudwatch';
import { SpudConfig, SpudConfigOpts } from './config';

export default function makeProcessor(spudConfig: SpudConfigOpts) {
  const config = new SpudConfig(spudConfig)

  async function processRecords(event: any, context: any): Promise<string> {
    console.info(`Received ${event.Records.length} events`);

    // Parse raw data into objects.
    const startTime = now();

    const logAggregator = new LogAggregator()

    let spadeEvents: SpadeEvent[] = [];
    for (let record of event.Records) {
      let e = new SpadeEvent(config, logAggregator);
      await e.populateDataPoints(record.kinesis.data);
      spadeEvents.push(e)
    }

    const samples: Sample[] = [].concat.apply([], spadeEvents.map((event: SpadeEvent) => event.toSamples()));
    const doneParsing = now();

    let agg = new Aggregator(60000);
    samples.forEach(s => agg.aggregateSample(s))

    const doneAggregating = now();

    let cwCollection = SpudCloudWatch.constructMetricDatums(agg.flush())

    await SpudCloudWatch.send(config.namespace, cwCollection)

    const completeTime = now();

    logAggregator.flush()

    console.info(`Parsing: ${(doneParsing - startTime).toFixed(3)}ms.\nAggregating: ${(doneAggregating - doneParsing).toFixed(3)}ms.\nCloudWatch: ${(completeTime - doneAggregating).toFixed(3)}ms.\nTotal: ${(completeTime - startTime).toFixed(3)}ms.`);

    return 'done';
  }

  return processRecords;
}
