import { Sample } from "./sample";

// bucketFactor is the configured bucket width.
// This should not be changed since it needs to match datastores that also use SEH1 histograms.
const BUCKET_FACTOR = Math.log(1 + 0.1);

// SEH1 is a sample aggregator that produces a sparse exponential histogram using an epsilon of 0.1.
export class SEH1 {
  // Histogram is a sparse exponential histogram representing the samples that were aggregated into this Datum.
  // It can be used to estimate percentiles.
  // Since SEH1 does not work for the value zero and negative values, these are counted using SEH1ZeroBucket.
  Histogram: Map<number, number> = new Map();

  // ZeroBucket is the bucket for counting zero values.
  ZeroBucket = 0;

  public constructor() { }

  // Include adds a sample to the histogram.
  public include(sample: Sample) {
    if (sample.Value > 0) {
      const bucket = this.computeBucket(sample.Value);
      if (!this.Histogram.get(bucket)) {
        this.Histogram.set(bucket, 1)
      } else {
        this.Histogram.set(bucket, (this.Histogram.get(bucket) as number) + 1)
      }
    } else {
      this.ZeroBucket++
    }
  }

  // ApproximateOriginalValue will give the estimated orignal value for items falling
  // into the given bucketIndex
  public approximateOriginalValue(bucketIndex: number): number {
    // compute originally --> y = math.Log(x) / bucketFactor
    // so... y*bucketFactor = math.Log(x)
    // so... math.Pow(e, y*bucketFactor) = x
    // Then add 0.5 to the bucket to make the value be in the center so that the
    // margin of error will average out and to avoid off by one errors
    // when converting bucket to value and back to bucket.
    return Math.pow(Math.E, (bucketIndex + 0.5) * BUCKET_FACTOR)
  }

  // helper to compute the bucket for a sample value
  private computeBucket(v: number): number {
    return Math.floor(Math.log(v) / BUCKET_FACTOR)
  }
}
