import moment = require('moment-timezone')

import { SpadeEventFields } from "./spade_event_data";
import { countryNameMap, regionMap } from "./geo";
import { SpudConfigData, MetricData } from './config';
import { Logger } from "./log-aggregator";
import { MetricID } from './metric_id';
import { Sample } from './sample';
import { filters } from './filters';

export interface DataPoint {
  config: SpudConfigData;
  fields: SpadeEventFields;
  toSamples(): Sample[];
}

export default class SpadeDataPoint implements DataPoint {
  config: SpudConfigData;
  logger: Logger;

  fields: SpadeEventFields;
  timestamp: number;

  constructor(readonly event_config: SpudConfigData, fields: SpadeEventFields, logger: Logger) {
    this.config = event_config;
    this.logger = logger;

    this.fields = fields;

    if (fields.time_utc) {
      this.timestamp = moment.tz(fields.time_utc, "Etc/UTC").toDate().getTime() * 1000;
    } else if (fields.time) {
      this.timestamp = moment.tz(fields.time, "America/Los_Angeles").tz("Etc/UTC").toDate().getTime() * 1000;
    } else {
      this.logger.log(`${this.config.event_name} does not contain a 'time_utc' or 'time' field, using current time.\nPlease add 'time_utc' your kinesis stream from Data Infra for accurate timestamps and aggregations`)
      this.timestamp = new Date().getTime() * 1000;
    }

    this.fields.country = countryNameMap[fields.country] ? countryNameMap[fields.country].name : `Unknown Country Code: ${fields.country}`;
    this.fields.region = regionMap[`${fields.country}:${fields.region}`] ? regionMap[`${fields.country}:${fields.region}`].region_name : '';
    this.fields.spud_event_counter = 1
  }

  public toSamples(): Sample[] {
    let sampleArr: Sample[] = [];

    for (let metricConf of this.config.metrics) {

      let f = Object.assign({}, this.fields);

      if (metricConf.alias) {
        for (let k of Object.keys(metricConf.alias).sort()) {
          f[k] = f[metricConf.alias[k]]
        }
      }

      if (metricConf.synthetic) {
        for (let k of Object.keys(metricConf.synthetic).sort()) {
          f[k] = metricConf.synthetic[k]
        }
      }

      let metricId = this.getMetricID(metricConf, f)
      if (metricId && filters.testAll(f, metricConf.filter, this.logger)) {
        let s: Sample = {
          MetricID: metricId,
          Timestamp: new Date(this.timestamp / 1000),
          Unit: metricConf.unit,
          Value: +f[metricConf.metric_field]
        }

        if (Number.isNaN(s.Value)) {
          this.logger.log('Invalid value for metric: ${this.config.event_name}');
          continue;
        }

        if (metricConf.rollup_dimensions) { s.RollupDimensions = metricConf.rollup_dimensions }
        sampleArr.push(s)
      }

    }
    return sampleArr
  }

  protected getMetricID(conf: MetricData, f: SpadeEventFields): MetricID | false {
    let id = new MetricID(conf.metric_name)
    if (conf.dimensions) {
      for (let dimension of conf.dimensions) {
        // If we expect a dimensions and it's missing/empty, discard the sample and log a message
        switch (f[dimension]) {
          case '':
          case undefined:
          case null:
            this.logger.log("Field for dimension '" + dimension + "' not found in spade data for '" + conf.metric_name + "'")
            return false;
          default:
            id.addDimension(dimension, f[dimension].toString())
        }
      }
    }
    return id
  }
}
