const { resolve } = require('path');
const schemaJson = require('./schema/schema.json');

const rollupPackages = ['pulsar', 'tachyon-utils'];

// maintain an allowlist here to make new additions more obvious
// (magic comments can't be used to disable react/no-danger)
const dangerouslySetInnerHTMLAllowlist = [
  'apps/starshot/src/components/pages/_document/SwitchHead.tsx',
  'packages/tachyon-core/latency-tracker/src/components/PerformancePolyfill.tsx',
  'packages/tachyon-core/more-ui/src/Fonts/index.tsx',
  'packages/tachyon-core/more-ui/src/tv/TvStyles.tsx',
  'packages/vx-prod/metadata/src/jsonld/index.tsx',
];

// Rule overrides for rules that are being enabled in phases across the repo
const slowRolloutOverrides = [
  // TODO: https://jira.xarth.tv/browse/EMP-3988
  {
    files: [
      'package-examples/client-environment-info/**',
      'package-examples/interaction-tracking/**',
      'packages/browser-clients/date-picker/**',
      'packages/tachyon-core/client-environment/**',
      'packages/tachyon-core/environment/**',
      'packages/tachyon-core/intl/**',
      'packages/tachyon-core/next-routing-utils/**',
      'packages/tachyon-core/relay/**',
      'packages/tachyon-core/test-utils/**',
    ],
    rules: {
      'react/prefer-stateless-function': 'off',
    },
  },
  {
    // roll out plan:
    // - everything under apps X
    // - laser array X
    // - packages
    // - everything
    files: ['apps/**/*.ts?(x)', 'native-apps/laser-array/**/*.ts?(x)'],
    excludedFiles: ['next-env.d.ts', '**/__mocks__/**'],
    parserOptions: {
      // https://github.com/typescript-eslint/typescript-eslint/blob/master/docs/getting-started/linting/MONOREPO.md
      project: process.env.LOCAL_TS
        ? // CLI/CI runs inside apps/packages via the node script with the env var
          './tsconfig.json'
        : // IDEs run from the root without the env var
          ['apps/*/tsconfig.json', 'native-apps/laser-array/tsconfig.json'],
    },
    extends: ['plugin:@typescript-eslint/recommended-requiring-type-checking'],
    rules: {
      '@typescript-eslint/no-confusing-void-expression': 'error',
      '@typescript-eslint/no-floating-promises': 'off', // so we don't need to handle router promises explicitly
      '@typescript-eslint/no-meaningless-void-operator': 'error',
      '@typescript-eslint/no-unnecessary-condition': 'error',
      '@typescript-eslint/no-unsafe-argument': 'error', // included in recommended config in next major
      '@typescript-eslint/prefer-includes': 'error',
      '@typescript-eslint/prefer-nullish-coalescing': 'error',
      '@typescript-eslint/prefer-reduce-type-parameter': 'error',
    },
  },
  // begin rolling out barrel-file enforcement
  // (eventually this should become a standard rule and not an override)
  {
    files: ['packages/vx-prod/discovery/**'],
    rules: {
      'import/no-internal-modules': 'off',
    },
  },
];

// Rule overrides for tests and test-related files
const testFileOverrides = [
  {
    files: ['*test.ts?(x)', '*test-utils.ts?(x)', '*test-mocks.ts?(x)'],
    rules: {
      // TS: allow non-null/any for convenience
      '@typescript-eslint/no-explicit-any': 'off',
      '@typescript-eslint/no-non-null-assertion': 'off',
      '@typescript-eslint/no-unsafe-argument': 'off',
      '@typescript-eslint/no-unsafe-assignment': 'off',
      '@typescript-eslint/no-unsafe-call': 'off',
      '@typescript-eslint/no-unsafe-member-access': 'off',
      '@typescript-eslint/no-unsafe-return': 'off',
      // react
      'react/display-name': 'off', // non-prod code doesn't need this
      'react/forbid-dom-props': 'off', // allow test files to use className for selector convenience
      'react/jsx-no-constructed-context-values': 'off', // non-prod code doesn't need this
    },
  },
  {
    files: ['*test.ts?(x)'],
    extends: ['plugin:jest/recommended', 'plugin:jest/style'],
    rules: {
      // jest specific tweaks
      'jest/consistent-test-it': ['error', { fn: 'it', withinDescribe: 'it' }],
      'jest/expect-expect': [
        'error',
        { assertFunctionNames: ['expect', 'expect*'] },
      ],
      'jest/no-alias-methods': 'error',
      'jest/no-disabled-tests': 'error',
      'jest/no-large-snapshots': [
        'error',
        {
          maxSize: 1,
        },
      ],
      'jest/valid-title': 'off', // we use the raw object as the test title
    },
  },
  {
    files: ['*test-utils.ts?(x)', '*test-mocks.ts?(x)'],
    rules: {
      // ignore at the file level since istanbul doesn't autorecognize these as test code
      'istanbul/no-ignore-file': 'off',
    },
  },
  {
    files: ['setupTests.ts'],
    rules: {
      '@typescript-eslint/no-unsafe-assignment': 'off', // for jest-fetch-mock
      'import/newline-after-import': 'off', // setupTests can have separated imports for logical grouping
    },
  },
  // cypress-specific rules
  {
    files: ['apps/*/cypress/**', 'native-apps/laser-array/cypress/**'],
    extends: ['plugin:cypress/recommended'],
    rules: {
      'import/no-internal-modules': [
        'error',
        {
          allow: [
            // cypress tests can import from config
            '**/src/config',
          ],
        },
      ],
      'no-unused-expressions': 'off', // Chai's BDD assertions look unused
    },
  },
];

// Rule overrides for the non-production code in examples and prototyper
const exampleOverrides = [
  {
    files: ['prototyper/**', 'package-examples/**'],
    rules: {
      'import/no-internal-modules': 'off',
      'no-alert': 'off',
      'no-console': 'off',
      'react/display-name': 'off',
    },
  },
];

const tomorrowOverrides = [
  {
    files: ['apps/tomorrow/**/*.tsx', 'packages/**/*.tsx'],
    rules: {
      // iOS devices don't support the autofocus property (it's a noop).
      // https://jira.twitch.com/browse/EMP-3116
      'jsx-a11y/no-autofocus': 'error',
    },
  },
];

const starshotOverrides = [
  {
    // allow pre-existing unstable components to stay until they are converted
    // https://jira.xarth.tv/browse/EMP-3930
    files: [
      'apps/starshot/src/components/pages/Homepage/index.tsx',
      'apps/starshot/src/components/pages/Search/SearchResults/index.tsx',
    ],
    rules: {
      'react/no-unstable-nested-components': 'warn',
    },
  },
];

module.exports = {
  extends: [
    'eslint:recommended',
    'plugin:@typescript-eslint/recommended',
    'plugin:import/typescript',
    'plugin:prettier/recommended',
    'plugin:react/recommended',
    'plugin:react/jsx-runtime',
    'plugin:react-hooks/recommended',
    'plugin:relay/strict',
    'plugin:tachyon/recommended',
    'plugin:typescript-sort-keys/recommended',
  ],
  plugins: [
    '@next/next',
    '@typescript-eslint',
    'cypress',
    'graphql',
    'import',
    'inclusive-language',
    'istanbul',
    'jest',
    'jsx-a11y',
    'react',
    'react-hooks',
    'relay',
    'sort-destructure-keys',
    'sort-keys-fix',
    'tachyon',
    'typescript-sort-keys',
    'twitch-core-ui',
  ],
  env: {
    browser: true,
    'jest/globals': true,
    node: true,
  },
  parser: '@typescript-eslint/parser',
  parserOptions: {
    ecmaVersion: 2019,
    ecmaFeatures: {
      impliedStrict: true,
      jsx: true,
    },
    warnOnUnsupportedTypeScriptVersion: true,
  },
  settings: {
    react: {
      version: 'detect',
    },
  },
  reportUnusedDisableDirectives: true,
  rules: {
    // typescript
    '@typescript-eslint/ban-types': [
      'error',
      {
        types: {
          '{}': false, // team preference
        },
      },
    ],
    '@typescript-eslint/consistent-type-imports': 'error', // for better compatibility with babel type-stripping
    '@typescript-eslint/explicit-module-boundary-types': 'off', // disable for all file types to avoid problems in JS
    '@typescript-eslint/no-empty-interface': 'off', // team preference
    '@typescript-eslint/no-explicit-any': 'off', // team preference
    '@typescript-eslint/no-non-null-assertion': 'error', // upgrade to error from default warning in recommended
    '@typescript-eslint/no-parameter-properties': 'error', // causes babel problems
    '@typescript-eslint/no-shadow': 'error', // preferred for its compatability with TS
    '@typescript-eslint/no-unused-vars': 'off', // TS handles this and ESLint doesn't understand _foo
    '@typescript-eslint/object-curly-spacing': ['error', 'always'], // preferred for its compatability with TS
    '@typescript-eslint/prefer-optional-chain': 'error', // team preference
    '@typescript-eslint/sort-type-union-intersection-members': 'error', // conforms to normal sorting preferences
    // graphql
    'graphql/no-deprecated-fields': [
      'error',
      {
        env: 'relay',
        tagName: 'graphql',
        schemaJson,
      },
    ],
    // import
    'import/first': 'error',
    'import/newline-after-import': 'error',
    'import/no-cycle': 'error',
    'import/no-duplicates': 'error', // preferred for its compatability with `import type ...` and autofix
    'import/no-default-export': 'error',
    'import/no-internal-modules': [
      'error',
      {
        allow: [
          // exempt relay type imports
          '*/__generated__/*',
          // Don't re-export mocks / test-utils with module index file to avoid
          // risk of accidental inclusion of large dev dependencies (Faker, Jest) in prod bundles
          '**/{test-utils,test-mocks}',
          // for next plugins
          '@next/*',
          'core-js/stable',
          // for next packages like server and link
          'next/*',
          // for various async test cases
          'react-dom/test-utils',
          'react-relay/legacy',
          'react-relay/hooks',
          // for sky-map code styling
          'react-syntax-highlighter/dist/cjs/styles/prism/*',
          'regenerator-runtime/runtime',
          // for various more-ui style rollups
          'tachyon-more-ui/styles/*',
        ],
      },
    ],
    'import/no-namespace': 'error',
    'import/no-relative-packages': 'error',
    'import/no-useless-path-segments': 'error',
    'import/order': [
      'error',
      {
        alphabetize: {
          order: 'asc',
        },
        'newlines-between': 'never',
      },
    ],
    // istanbul
    'istanbul/no-ignore-file': 'error',
    'istanbul/prefer-ignore-reason': 'error',
    // react
    'react/display-name': [2, { ignoreTranspilerName: true }],
    'react/forbid-dom-props': ['error', { forbid: ['className', 'style'] }],
    'react/jsx-boolean-value': 'error',
    'react/jsx-no-constructed-context-values': 'error',
    'react/jsx-no-useless-fragment': ['error', { allowExpressions: true }],
    'react/jsx-sort-props': 'error',
    'react/no-children-prop': 'off',
    'react/no-danger': 'error',
    'react/no-namespace': 'error',
    'react/no-unsafe': 'error',
    'react/no-unstable-nested-components': 'error',
    'react/prefer-stateless-function': 'error', // team preference
    'react/prop-types': 'off',
    'react/self-closing-comp': 'error',
    // react-hooks
    'react-hooks/exhaustive-deps': 'error',
    // relay
    'relay/generated-flow-types': 'off',
    'relay/must-colocate-fragment-spreads': 'off', // incompatible with current Next setup
    'relay/unused-fields': 'off',
    // eslint
    curly: 'error',
    eqeqeq: 'error',
    'guard-for-in': 'error',
    'key-spacing': [
      'error',
      { beforeColon: false, afterColon: true, mode: 'strict' },
    ],
    'keyword-spacing': ['error', { before: true, after: true }],
    'line-comment-position': 'error',
    'new-cap': 'error',
    'no-alert': 'error',
    'no-case-declarations': 'off',
    'no-console': 'error', // disable console to force use of logger
    'no-duplicate-imports': 'off', // prefer `@typescript-eslint/no-duplicate-imports` for `import type ...` compatability
    'no-else-return': 'error',
    'no-eval': 'error',
    'no-floating-decimal': 'error',
    'no-implicit-coercion': ['error', { boolean: false }],
    'no-implied-eval': 'error',
    'no-label-var': 'error',
    'no-octal-escape': 'error',
    'no-restricted-globals': ['error', 'xdescribe', 'fit', 'fdescribe'],
    'no-shadow': 'off', // prefer `@typescript-eslint/no-shadow` for compatability with TS
    'no-template-curly-in-string': 'error',
    'no-throw-literal': 'error',
    'no-unused-expressions': ['error', { enforceForJSX: true }],
    'no-useless-computed-key': 'error',
    'object-curly-spacing': 'off', // prefer `@typescript-eslint/object-curly-spacing` for compatability with TS
    'object-shorthand': ['error', 'always'],
    'prefer-const': 'error',
    'prefer-object-spread': 'error',
    'prefer-template': 'error',
    'quote-props': ['error', 'as-needed'],
    'sort-imports': ['error', { ignoreDeclarationSort: true }],
    'spaced-comment': ['error', 'always', { markers: ['/ <reference'] }],
    'symbol-description': 'error',
    'valid-typeof': ['error', { requireStringLiterals: true }],
    // sort destructure
    'sort-destructure-keys/sort-destructure-keys': [
      'error',
      { caseSensitive: true },
    ],
    // sort keys
    'sort-keys-fix/sort-keys-fix': ['error', 'asc', { natural: true }], // autofix version of the builtin sort-keys
    // Inclusive
    'inclusive-language/use-inclusive-words': [
      'error',
      {
        allowedTerms: [
          // allow links to github
          { term: 'blob/master', allowPartialMatches: true },
          // allow references to dev gql
          { term: 'master/gql', allowPartialMatches: true },
          // allow error term from player-core
          { term: 'masterplaylist' },
        ],
        lintStrings: true,
      },
    ],
    // Core UI
    'twitch-core-ui/no-styled-layout': 'error',
    // Tachyon
    'tachyon/bowser-parser-fallback': 'error',
    'tachyon/no-aliased-absolute-imports': ['error', ['src']],
    'tachyon/no-disable-rules': [
      'error',
      {
        noEmptyDisables: true,
        rules: [
          // push people into @ts-expect-error
          '@typescript-eslint/ban-ts-comment',
          // per security policy
          'react/no-danger',
        ],
      },
    ],
    'tachyon/no-single-child-directional-navs': 'error',
  },
  overrides: [
    // slow rollout must come first to simulate future config
    ...slowRolloutOverrides,
    ...testFileOverrides,
    ...exampleOverrides,
    ...tomorrowOverrides,
    ...starshotOverrides,
    // enable rule specifically for TypeScript files
    // rm after everything is opted into 'plugin:@typescript-eslint/recommended-requiring-type-checking'
    {
      files: ['*.ts', '*.tsx'],
      rules: {
        '@typescript-eslint/explicit-module-boundary-types': ['error'],
      },
    },
    // no rollup imports in packages
    {
      files: ['packages/**/*'],
      rules: {
        'tachyon/no-rollup-package-import': ['error', rollupPackages],
      },
    },
    // no sub-package imports in apps or examples
    {
      files: ['apps/**/*', 'native-apps/**/*', 'package-examples/**/*'],
      rules: {
        'tachyon/no-sub-package-import': ['error', rollupPackages],
      },
    },
    // allow requires in non-transpiled JS files and logical key ordering in config files
    {
      files: [
        '.eslintrc.js',
        'babel-node.js',
        '*babel.config.js',
        'commitlint.config.js',
        'env.config.js',
        'jest.config.js',
        'next.config.js',
        'webpack.config.js',
        'packages/tachyon-core/package-builder/**',
        'jest.config/*.js',
      ],
      rules: {
        '@typescript-eslint/no-var-requires': 'off',
        'import/no-internal-modules': 'off',
        'line-comment-position': 'off',
        'sort-keys-fix/sort-keys-fix': 'off',
      },
    },
    // allow console usage in places where logger is inappropriate
    {
      files: [
        'apps/*/src/server/**',
        'apps/*/server/**',
        'apps/*/scripts/**',
        'packages/tachyon-core/package-builder/**',
        'scripts/**',
      ],
      rules: {
        'no-console': 'off',
      },
    },
    // next pages must have a default export
    {
      files: [
        'apps/moonbase/src/pages/**',
        'apps/starshot/src/pages/**',
        'apps/sky-map/src/pages/**',
        'apps/tomorrow/src/pages/**',
        'apps/valence/src/pages/**',
      ],
      rules: {
        'import/no-default-export': 'off',
      },
    },
    // enable next lint plugin for next apps
    {
      files: ['apps/**'],
      extends: [
        'plugin:@next/next/recommended',
        'plugin:@next/next/core-web-vitals',
      ],
    },
    // disable barrel enforcement for JS packages/scripts
    {
      files: [
        'packages/tachyon-core/eslint-plugin/**',
        'packages/tachyon-core/package-builder/**',
        'packages/tachyon-core/relay/updateSchema.js',
      ],
      rules: {
        'import/no-internal-modules': 'off',
      },
    },
    // allowed usages of dangerouslySetInnerHTML
    {
      files: dangerouslySetInnerHTMLAllowlist,
      rules: {
        // left as warn to highlight danger
        'react/no-danger': 'warn',
      },
    },
  ],
};
