import jetbrains.buildServer.configs.kotlin.v2018_2.ParameterDisplay
import jetbrains.buildServer.configs.kotlin.v2018_2.ReuseBuilds
import jetbrains.buildServer.configs.kotlin.v2018_2.buildSteps.exec
import jetbrains.buildServer.configs.kotlin.v2018_2.buildSteps.script
import jetbrains.buildServer.configs.kotlin.v2018_2.FailureAction
import jetbrains.buildServer.configs.kotlin.v2018_2.version
import jetbrains.buildServer.configs.kotlin.v2018_2.Requirements
// to download instrumentorum as a maven dependency, you will need to be on TwitchVPN
import twitch.instrumentorum.enums.AWSRegion
import twitch.instrumentorum.enums.BranchFilter
import twitch.instrumentorum.instrum
import twitch.instrumentorum.project.builds.dockerAwareBuild
import twitch.instrumentorum.project.builds.dockerBuildAndPush
import twitch.instrumentorum.project.builds.initializeBuildChain
import twitch.instrumentorum.project.builds.updateECSService
import twitch.instrumentorum.project.steps.verifyECRRepository

// TeamCity API: https://teamcity.jetbrains.com/app/dsl-documentation/jetbrains.build-server.configs.kotlin.v2018_2/index.html
// relevant intrumentorum source: https://git.xarth.tv/CPE-Ops/instrumentorum/tree/master/src/code/project

// specify the TeamCity DSL version
version = "2018.2"

instrum {
    description = "CI for the Tachyon Monorepo"

    textParam("git.branch.default", "refs/heads/main")
    sourceRoot("git@git.xarth.tv:emerging-platforms/tachyon.git")
    settingsRoot("git@git.xarth.tv:emerging-platforms/tachyon.git")

    textParam("ecr.repo", "tachyon", "ECR Repository", "The name of the ECR Repository used for deploys", readOnly = true)
    textParam("ecr.repo.dependencies", "tachyon-build-dependencies", "ECR Repository", "The name of the ECR Repository used for intermediary container(s)", readOnly = true)

    textParam("arn.upload-role", "arn:aws:iam::015957721237:role/TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018", "AWS Role ARN", "ARN for the role to upload to S3", readOnly = true)
    textParam("bucket.js1", "mweb-assets-production-useast2-fbc9c735a45f475be19d9c9a6f3b2a5c", "S3 Bucket", "Name of the S3 bucket for JS assets", readOnly = true)
    textParam("bucket.js2", "mweb-assets-production-uswest2-6bff4b7ae46fe79051edfbee0904243c", "S3 Bucket", "Name of the S3 bucket for JS assets", readOnly = true)
    textParam("bucket.js3", "mweb-assets-production-apsoutheast1-81d5c790ea8b77812c0a1b5921b", "S3 Bucket", "Name of the S3 bucket for JS assets", readOnly = true)
    textParam("bucket.js4", "mweb-assets-production-eucentral1-f893da78663c1db65c6ce061525fd", "S3 Bucket", "Name of the S3 bucket for JS assets", readOnly = true)

    subInstrum("Prepare Dependencies and Packages", "Prepare") {
      // initializeBuildChain creates a BuildType that performs common, sane steps to help with identifying builds
      initializeBuildChain {
          // `ecr.domain` is declared inside steps/builds, so we cannot reference it globally (aka in this step)

          // ensure that the ECR repository for our intermediary build/test image exists (why do we do this?)
          verifyECRRepository("%ecr.repo.dependencies%")

          // IMPORTANT!!
          // Once we move to the new infra, the tags should be {app/stage}-{gitSHA}
          // ECR allows filtering by prefix, and not by suffix

          textParam("docker.tag.prepare", "%git.commit.short%-prepare", "Tag for Prepare Image", display = ParameterDisplay.HIDDEN)
          textParam("docker.image.prepare", "%ecr.repo.dependencies%:%docker.tag.prepare%", "URL for Prepare Image", display = ParameterDisplay.HIDDEN)

          textParam("docker.tag.packages", "%git.commit.short%-packages", "Tag for Packages Image", display = ParameterDisplay.HIDDEN)
          textParam("docker.image.packages", "%ecr.repo.dependencies%:%docker.tag.packages%", "URL for Packages Image", display = ParameterDisplay.HIDDEN)

          textParam("docker.tag.tomorrow", "%git.commit.short%-tomorrow", "Tag for Tomorrow Production Image", display = ParameterDisplay.HIDDEN)
          textParam("docker.image.tomorrow", "%ecr.repo%:%docker.tag.tomorrow%", "URL for Tomorrow Production Image", display = ParameterDisplay.HIDDEN)

          textParam("docker.tag.starshot", "%git.commit.short%-starshot", "Docker Tag for Starshot Production Image", display = ParameterDisplay.HIDDEN)
          textParam("docker.image.starshot", "%ecr.repo%:%docker.tag.starshot%", "URL for Starshot Production Image", display = ParameterDisplay.HIDDEN)

          textParam("docker.tag.valence", "%git.commit.short%-valence", "Docker Tag for Valence Production Image", display = ParameterDisplay.HIDDEN)
          textParam("docker.image.valence", "%ecr.repo%:%docker.tag.valence%", "URL for Valence Production Image", display = ParameterDisplay.HIDDEN)

          textParam("docker.tag.moonbase", "%git.commit.short%-moonbase", "Docker Tag for Moonbase Production Image", display = ParameterDisplay.HIDDEN)
          textParam("docker.image.moonbase", "%ecr.repo%:%docker.tag.moonbase%", "URL for Moonbase Production Image", display = ParameterDisplay.HIDDEN)

          textParam("docker.tag.android", "%git.commit.short%-android", "Docker Tag for Android/FireTV SDK Image", display = ParameterDisplay.HIDDEN)
          textParam("docker.image.android", "%ecr.repo%:%docker.tag.android%", "URL for Android/FireTV SDK Image", display = ParameterDisplay.HIDDEN)
      }

      // create an initial image with all dependencies
      dockerBuildAndPush("Prepare Monorepo") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          textParam("env.DOCKER_BUILDKIT", "1", display = ParameterDisplay.HIDDEN)
          requirements {
            noLessThanVer("docker.version", "19.03")
          }

          ecrRepository = "%ecr.repo.dependencies%"
          tag = paramFrom("Initialize Build Chain", "docker.tag.prepare")
          targetStage = "prepare-monorepo"
      }

      dockerBuildAndPush("Build Packages") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          paramFrom("Initialize Build Chain", "docker.image.prepare")
          textParam("env.GIT_COMMIT", valueFrom("Initialize Build Chain", "git.commit.short"), display = ParameterDisplay.HIDDEN)
          textParam("env.DOCKER_BUILDKIT", "1", display = ParameterDisplay.HIDDEN)
          requirements {
            noLessThanVer("docker.version", "19.03")
          }

          snapshotDependency("Prepare", "Prepare Monorepo")

          githubStatusPublisher()

          ecrRepository = "%ecr.repo.dependencies%"
          tag = paramFrom("Initialize Build Chain", "docker.tag.packages")
          targetStage = "build-packages"
          cacheFrom = "%ecr.domain%/%docker.image.prepare%"
      }
    }

    subInstrum("Lint and Analyze", "Lint") {
        // even though builds will currently fail if there are type errors, this also catches type errors in tests
        dockerAwareBuild("Typecheck") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")

            snapshotDependency("Prepare", "Build Packages")

            githubStatusPublisher()

            steps {
                script {
                    name = "Run tsc"
                    scriptContent = """
                        #!/usr/bin/env bash
                        cd /opt/tachyon
                        yarn lerna run typecheck
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }
            }
        }

        dockerAwareBuild("Lint TypeScript") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")

            snapshotDependency("Lint", "Typecheck") {
              runOnSameAgent = true
              onDependencyFailure = FailureAction.IGNORE
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Run eslint"
                    scriptContent = """
                        #!/usr/bin/env bash
                        cd /opt/tachyon
                        yarn lint:es:ci
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }
            }
        }

        dockerAwareBuild("Lint Styles") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.prepare")

            snapshotDependency("Prepare", "Prepare Monorepo")

            githubStatusPublisher()

            steps {
                script {
                    name = "Run stylelint"
                    scriptContent = """
                        #!/usr/bin/env bash
                        cd /opt/tachyon
                        yarn lint:styles:ci
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.prepare%"
                }
            }
        }

        dockerAwareBuild("Lint Markdown") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.prepare")

            snapshotDependency("Lint", "Lint Styles") {
              runOnSameAgent = true
              onDependencyFailure = FailureAction.IGNORE
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Run prettier on markdown"
                    scriptContent = """
                        #!/usr/bin/env bash
                        cd /opt/tachyon
                        yarn lint:md
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.prepare%"
                }
            }
        }

        dockerAwareBuild("Lint package.json") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.prepare")

            snapshotDependency("Lint", "Lint Markdown") {
              runOnSameAgent = true
              onDependencyFailure = FailureAction.IGNORE
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Run prettier-package-json on package.json"
                    scriptContent = """
                        #!/usr/bin/env bash
                        cd /opt/tachyon
                        yarn lint:package:ci
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.prepare%"
                }
            }
        }

        dockerAwareBuild("Verify i18n") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")

            snapshotDependency("Prepare", "Build Packages")

            githubStatusPublisher()

            steps {
                script {
                    name = "Run twitch-intl submission-check"
                    scriptContent = """
                        #!/usr/bin/env bash
                        cd /opt/tachyon
                        yarn lerna run intl:test:ci
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }
            }
        }
    }

    subInstrum("Test Packages and Apps", "Test") {
        dockerAwareBuild("Package Tests") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.CODECOV_TOKEN", "credentialsJSON:e42ea517-4865-4432-b8ee-4e3dbc1b1ee8", "Codecov upload token")

            snapshotDependency("Prepare", "Build Packages")

            failureConditions {
                executionTimeoutMin = 10 // fail if tests run longer than 10 minutes
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Run Packages' Jest"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir coverage
                        export CODE_COV_DEST_DIR=$(pwd)/coverage
                        cd /opt/tachyon
                        yarn test:packages:ci
                        cp jest.config/test-results/coverage-final.json ${'$'}CODE_COV_DEST_DIR
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Report to CodeCov"
                    scriptContent = """
                        #!/usr/bin/env bash
                        curl -s https://codecov.xarth.tv/bash | bash -s - -t ${'$'}CODECOV_TOKEN -s coverage
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Tomorrow Tests") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.CODECOV_TOKEN", "credentialsJSON:e42ea517-4865-4432-b8ee-4e3dbc1b1ee8", "Codecov upload token")

            snapshotDependency("Prepare", "Build Packages")

            failureConditions {
                executionTimeoutMin = 10 // fail if tests run longer than 10 minutes
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Ensure Relay Query Types Updated"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        cd /opt/tachyon/apps/tomorrow
                        if [[ `yarn relay | grep Updated` ]]; then
                            echo "The relay generated query types need to be updated. Run \`yarn relay\`."
                            exit 1
                        fi
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Run Tomorrow Jest"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir coverage
                        export CODE_COV_DEST_DIR=$(pwd)/coverage
                        cd /opt/tachyon
                        yarn test:app:tmw:ci
                        cp jest.config/test-results/coverage-final.json ${'$'}CODE_COV_DEST_DIR
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Report to CodeCov"
                    scriptContent = """
                        #!/usr/bin/env bash
                        curl -s https://codecov.xarth.tv/bash | bash -s - -t ${'$'}CODECOV_TOKEN -s coverage
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Starshot Tests") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.CODECOV_TOKEN", "credentialsJSON:e42ea517-4865-4432-b8ee-4e3dbc1b1ee8", "Codecov upload token")

            snapshotDependency("Prepare", "Build Packages")

            failureConditions {
                executionTimeoutMin = 10 // fail if tests run longer than 10 minutes
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Ensure Relay Query Types Updated"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        cd /opt/tachyon/apps/starshot
                        if [[ `yarn relay | grep Updated` ]]; then
                            echo "The relay generated query types need to be updated. Run \`yarn relay\`."
                            exit 1
                        fi
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Run Starshot Jest"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir coverage
                        export CODE_COV_DEST_DIR=$(pwd)/coverage
                        cd /opt/tachyon
                        yarn test:app:sst:ci
                        cp jest.config/test-results/coverage-final.json ${'$'}CODE_COV_DEST_DIR
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Report to CodeCov"
                    scriptContent = """
                        #!/usr/bin/env bash
                        curl -s https://codecov.xarth.tv/bash | bash -s - -t ${'$'}CODECOV_TOKEN -s coverage
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Valence Tests") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.CODECOV_TOKEN", "credentialsJSON:e42ea517-4865-4432-b8ee-4e3dbc1b1ee8", "Codecov upload token")

            snapshotDependency("Prepare", "Build Packages")

            failureConditions {
                executionTimeoutMin = 10 // fail if tests run longer than 10 minutes
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Ensure Relay Query Types Updated"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        cd /opt/tachyon/apps/valence
                        if [[ `yarn relay | grep Updated` ]]; then
                            echo "The relay generated query types need to be updated. Run \`yarn relay\`."
                            exit 1
                        fi
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Run Valence Jest"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir coverage
                        export CODE_COV_DEST_DIR=$(pwd)/coverage
                        cd /opt/tachyon
                        yarn test:app:vlc:ci
                        cp jest.config/test-results/coverage-final.json ${'$'}CODE_COV_DEST_DIR
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Report to CodeCov"
                    scriptContent = """
                        #!/usr/bin/env bash
                        curl -s https://codecov.xarth.tv/bash | bash -s - -t ${'$'}CODECOV_TOKEN -s coverage
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Moonbase Tests") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.CODECOV_TOKEN", "credentialsJSON:e42ea517-4865-4432-b8ee-4e3dbc1b1ee8", "Codecov upload token")

            snapshotDependency("Prepare", "Build Packages")

            failureConditions {
                executionTimeoutMin = 10 // fail if tests run longer than 10 minutes
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Ensure Relay Query Types Updated"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        cd /opt/tachyon/apps/moonbase
                        if [[ `yarn relay | grep Updated` ]]; then
                            echo "The relay generated query types need to be updated. Run \`yarn relay\`."
                            exit 1
                        fi
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Run Moonbase Jest"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir coverage
                        export CODE_COV_DEST_DIR=$(pwd)/coverage
                        cd /opt/tachyon
                        yarn test:app:mnb:ci
                        cp jest.config/test-results/coverage-final.json ${'$'}CODE_COV_DEST_DIR
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Report to CodeCov"
                    scriptContent = """
                        #!/usr/bin/env bash
                        curl -s https://codecov.xarth.tv/bash | bash -s - -t ${'$'}CODECOV_TOKEN -s coverage
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Sky Map Tests") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.CODECOV_TOKEN", "credentialsJSON:e42ea517-4865-4432-b8ee-4e3dbc1b1ee8", "Codecov upload token")

            snapshotDependency("Prepare", "Build Packages")

            failureConditions {
                executionTimeoutMin = 10 // fail if tests run longer than 10 minutes
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Run Sky Map Jest"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir coverage
                        export CODE_COV_DEST_DIR=$(pwd)/coverage
                        cd /opt/tachyon
                        yarn test:app:sky:ci
                        cp jest.config/test-results/coverage-final.json ${'$'}CODE_COV_DEST_DIR
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Report to CodeCov"
                    scriptContent = """
                        #!/usr/bin/env bash
                        curl -s https://codecov.xarth.tv/bash | bash -s - -t ${'$'}CODECOV_TOKEN -s coverage
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Laser Array Tests") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.CODECOV_TOKEN", "credentialsJSON:e42ea517-4865-4432-b8ee-4e3dbc1b1ee8", "Codecov upload token")

            snapshotDependency("Prepare", "Build Packages")

            failureConditions {
                executionTimeoutMin = 10 // fail if tests run longer than 10 minutes
            }

            githubStatusPublisher()

            steps {
                script {
                    name = "Run Laser Array Jest"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir coverage
                        export CODE_COV_DEST_DIR=$(pwd)/coverage
                        cd /opt/tachyon
                        yarn test:app:lsr:ci
                        cp jest.config/test-results/coverage-final.json ${'$'}CODE_COV_DEST_DIR
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Report to CodeCov"
                    scriptContent = """
                        #!/usr/bin/env bash
                        curl -s https://codecov.xarth.tv/bash | bash -s - -t ${'$'}CODECOV_TOKEN -s coverage
                    """.trimIndent()
                }
            }
        }
    }

    subInstrum("Build App Containers", "Build") {
        dockerBuildAndPush("Build Tomorrow") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.SENTRY_AUTH_TOKEN", "credentialsJSON:6f29867a-30e5-4171-9386-76e506674e50", "Sentry upload token")
            textParam("env.GIT_COMMIT", valueFrom("Initialize Build Chain", "git.commit.short"), display = ParameterDisplay.HIDDEN)
            textParam("env.DOCKER_BUILDKIT", "1", display = ParameterDisplay.HIDDEN)
            requirements {
              noLessThanVer("docker.version", "19.03")
            }

            snapshotDependency("Prepare", "Build Packages")

            githubStatusPublisher()

            dockerArgs = "--build-arg GIT_COMMIT --build-arg SENTRY_AUTH_TOKEN"
            tag = paramFrom("Initialize Build Chain", "docker.tag.tomorrow")
            targetStage = "tomorrow-production"
            cacheFrom = "%ecr.domain%/%docker.image.packages%"
        }

        dockerBuildAndPush("Build Starshot") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.SENTRY_AUTH_TOKEN", "credentialsJSON:6f29867a-30e5-4171-9386-76e506674e50", "Sentry upload token")
            textParam("env.GIT_COMMIT", valueFrom("Initialize Build Chain", "git.commit.short"), display = ParameterDisplay.HIDDEN)
            textParam("env.DOCKER_BUILDKIT", "1", display = ParameterDisplay.HIDDEN)
            requirements {
              noLessThanVer("docker.version", "19.03")
            }

            snapshotDependency("Prepare", "Build Packages")

            githubStatusPublisher()

            dockerArgs = "--build-arg GIT_COMMIT --build-arg SENTRY_AUTH_TOKEN"
            tag = paramFrom("Initialize Build Chain", "docker.tag.starshot")
            targetStage = "starshot-production"
            cacheFrom = "%ecr.domain%/%docker.image.packages%"
        }

        dockerBuildAndPush("Build Valence") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.SENTRY_AUTH_TOKEN", "credentialsJSON:6f29867a-30e5-4171-9386-76e506674e50", "Sentry upload token")
            textParam("env.GIT_COMMIT", valueFrom("Initialize Build Chain", "git.commit.short"), display = ParameterDisplay.HIDDEN)
            textParam("env.DOCKER_BUILDKIT", "1", display = ParameterDisplay.HIDDEN)
            requirements {
              noLessThanVer("docker.version", "19.03")
            }

            snapshotDependency("Prepare", "Build Packages")

            githubStatusPublisher()

            dockerArgs = "--build-arg GIT_COMMIT --build-arg SENTRY_AUTH_TOKEN"
            tag = paramFrom("Initialize Build Chain", "docker.tag.valence")
            targetStage = "valence-production"
            cacheFrom = "%ecr.domain%/%docker.image.packages%"
        }

        dockerBuildAndPush("Build Moonbase") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")
            passwordParam("env.SENTRY_AUTH_TOKEN", "credentialsJSON:6f29867a-30e5-4171-9386-76e506674e50", "Sentry upload token")
            textParam("env.GIT_COMMIT", valueFrom("Initialize Build Chain", "git.commit.short"), display = ParameterDisplay.HIDDEN)
            textParam("env.DOCKER_BUILDKIT", "1", display = ParameterDisplay.HIDDEN)
            requirements {
              noLessThanVer("docker.version", "19.03")
            }

            snapshotDependency("Prepare", "Build Packages")

            githubStatusPublisher()

            dockerArgs = "--build-arg GIT_COMMIT --build-arg SENTRY_AUTH_TOKEN"
            tag = paramFrom("Initialize Build Chain", "docker.tag.moonbase")
            targetStage = "moonbase-production"
            cacheFrom = "%ecr.domain%/%docker.image.packages%"
        }
    }

    subInstrum("Tachyon Staging", "Staging") {
      updateECSService("Deploy App to Dev Pool") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          selectParam("tachyon.app", tachyonApps[0], "Tachyon App", options = tachyonApps, display = ParameterDisplay.PROMPT)
          selectParam("dev.ecs.service", devPoolSlots[0], "Tachyon Dev Pool Slot", options = devPoolSlots, display = ParameterDisplay.PROMPT)
          // for now only Starshot uses the platform env var
          selectParam("tachyon.platform", "staging_web_tv", "Tachyon Platform", options = tachyonPlatforms, display = ParameterDisplay.PROMPT)

          snapshotDependency("Build", "Build Tomorrow")
          snapshotDependency("Build", "Build Starshot")
          snapshotDependency("Build", "Build Valence")
          snapshotDependency("Build", "Build Moonbase")

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%ecr.repo%:%git.commit.short%-%tachyon.app%"
          ecsAccountId = "111910252139"
          ecsCluster = "TwitchEmpDevFargate-servicesC9B6BABA-NFsLagWhpAqA"
          ecsService = "EmpFargateService%dev.ecs.service%"

          deployRole = "TeamcityToECSIAMRole-USWest2"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          steps {
              script {
                  name = "Update Build Number"
                  scriptContent = """
                  #!/usr/bin/env bash
                  echo '##teamcity[buildNumber' "'"${'$'}BUILD_NUMBER'-Dev%dev.ecs.service%'"'"]
              """.trimIndent()
              }
          }

          updateRegion(AWSRegion.PDX, "tasks/dev.json", 1)
      }

      updateECSService("Deploy App to QA Pool") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          selectParam("tachyon.app", tachyonApps[0], "Tachyon App", options = tachyonApps, display = ParameterDisplay.PROMPT)
          selectParam("qa.ecs.service", qaPoolSlots[0], "Tachyon QA Pool Slot", options = qaPoolSlots, display = ParameterDisplay.PROMPT)
          // for now only Starshot uses the platform env var
          selectParam("tachyon.platform", "staging_web_tv", "Tachyon Platform", options = tachyonPlatforms, display = ParameterDisplay.PROMPT)

          snapshotDependency("Build", "Build Tomorrow")
          snapshotDependency("Build", "Build Starshot")
          snapshotDependency("Build", "Build Valence")
          snapshotDependency("Build", "Build Moonbase")

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%ecr.repo%:%git.commit.short%-%tachyon.app%"
          ecsAccountId = "820097635994"
          ecsCluster = "EmpEcsServiceCluster"
          ecsService = "%qa.ecs.service%EcsService"

          deployRole = "TeamcityToECSIAMRole-USWest2"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          steps {
              script {
                  name = "Update Build Number"
                  scriptContent = """
                  #!/usr/bin/env bash
                  echo '##teamcity[buildNumber' "'"${'$'}BUILD_NUMBER'-%qa.ecs.service%'"'"]
              """.trimIndent()
              }
          }

          updateRegion(AWSRegion.PDX, "tasks/qa.json", 1)
      }

      updateECSService("Tomorrow Main QA") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          paramFrom("Initialize Build Chain", "docker.image.tomorrow")
          textParam("tachyon.app", "tomorrow")
          textParam("qa.ecs.service", "sdf29i2nofmnfsdf0sdf2msz")
          textParam("tachyon.platform", "mobile_web")

          // this should match the "Tomorrow Production" list to pre-cache these reqs
          snapshotDependency("Lint", "Typecheck")
          snapshotDependency("Test", "Package Tests")
          snapshotDependency("Test", "Tomorrow Tests")
          snapshotDependency("Build", "Build Tomorrow")

          // auto-triggers for main only
          vcsTrigger {
              branchFilter = """
                  +:<default>
              """.trimIndent()
          }

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
          ecsAccountId = "820097635994"
          ecsCluster = "EmpEcsServiceCluster"
          ecsService = "%qa.ecs.service%EcsService"

          deployRole = "TeamcityToECSIAMRole-USWest2"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          steps {
              script {
                  name = "Filter Non-Main Deploys"
                  scriptContent = """
                      #!/usr/bin/env bash
                      if [ "%teamcity.build.branch%" != "main" ]; then
                          echo ##teamcity[buildProblem description='QA %qa.ecs.service% slot is reserved for "main" releases only']
                          exit 1
                      fi
                  """.trimIndent()
              }
          }

          updateRegion(AWSRegion.PDX, "tasks/qa.json", 1)
      }

      updateECSService("Starshot Main QA") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          paramFrom("Initialize Build Chain", "docker.image.starshot")
          textParam("tachyon.app", "starshot")
          textParam("qa.ecs.service", "isodfj9238hfwiodnfskuvdf")
          textParam("tachyon.platform", "staging_web_tv")

          // this should match the "Starshot Production" list to pre-cache these reqs
          snapshotDependency("Lint", "Typecheck")
          snapshotDependency("Test", "Package Tests")
          snapshotDependency("Test", "Starshot Tests")
          snapshotDependency("Build", "Build Starshot")

          // auto-triggers for main only
          vcsTrigger {
              branchFilter = """
                  +:<default>
              """.trimIndent()
          }

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%docker.image.starshot%"
          ecsAccountId = "820097635994"
          ecsCluster = "EmpEcsServiceCluster"
          ecsService = "%qa.ecs.service%EcsService"

          deployRole = "TeamcityToECSIAMRole-USWest2"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          steps {
              script {
                  name = "Filter Non-Main Deploys"
                  scriptContent = """
                      #!/usr/bin/env bash
                      if [ "%teamcity.build.branch%" != "main" ]; then
                          echo ##teamcity[buildProblem description='QA %qa.ecs.service% slot is reserved for "main" releases only']
                          exit 1
                      fi
                  """.trimIndent()
              }
          }

          updateRegion(AWSRegion.PDX, "tasks/qa.json", 1)
      }

      updateECSService("Tomorrow VP QA") {
        paramFrom("Initialize Build Chain", "git.commit.short")
        paramFrom("Initialize Build Chain", "docker.image.tomorrow")
        textParam("tachyon.app", "tomorrow")
        textParam("qa.ecs.service", "xfelfmacdvxakc6ooafsgkv9")
        textParam("tachyon.platform", "mobile_web")

        snapshotDependency("Build", "Build Tomorrow")

        // translates to `docker.image` in task via instrumentorum
        dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
        ecsAccountId = "820097635994"
        ecsCluster = "EmpEcsServiceCluster"
        ecsService = "%qa.ecs.service%EcsService"

        deployRole = "TeamcityToECSIAMRole-USWest2"

        failureConditions {
          executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
        }

        updateRegion(AWSRegion.PDX, "tasks/qa.json", 1)
      }

      updateECSService("Deploy App to Environment (old pool)") {
          paramFrom("Initialize Build Chain", "git.commit.short")

          selectParam("tachyon.app", tachyonApps[0], "Tachyon App", options = tachyonApps, display = ParameterDisplay.PROMPT)
          selectParam("staging.ecs.service", starshotPoolSlots[0], "Tachyon Staging Slot", options = starshotPoolSlots, display = ParameterDisplay.PROMPT)
          // for now only Starshot uses the platform env var
          selectParam("tachyon.platform", "staging_web_tv", "Tachyon Platform", options = tachyonPlatforms, display = ParameterDisplay.PROMPT)

          snapshotDependency("Build", "Build Tomorrow")
          snapshotDependency("Build", "Build Starshot")
          snapshotDependency("Build", "Build Valence")
          snapshotDependency("Build", "Build Moonbase")

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%ecr.repo%:%git.commit.short%-%tachyon.app%"
          ecsAccountId = "709603242682"
          ecsCluster = "starshot"
          ecsService = "%staging.ecs.service%"

          deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          steps {
              script {
                  name = "Update Build Number"
                  scriptContent = """
                  #!/usr/bin/env bash
                  echo '##teamcity[buildNumber' "'"${'$'}BUILD_NUMBER'-%staging.ecs.service%'"'"]
              """.trimIndent()
              }
          }

          updateRegion(AWSRegion.PDX, "tasks/staging.json", 1)
      }

      updateECSService("Deploy TMW for VP QA (old pool)") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          paramFrom("Initialize Build Chain", "docker.image.tomorrow")
          textParam("tachyon.app", "tomorrow")
          textParam("tachyon.platform", "mobile_web")

          snapshotDependency("Build", "Build Tomorrow")

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
          ecsAccountId = "709603242682"
          ecsCluster = "starshot"
          ecsService = "dmwhop68wex46a485ze6"

          deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          updateRegion(AWSRegion.PDX, "tasks/staging.json", 1)
      }

      updateECSService("Tomorrow Main Staging (old pool)") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          paramFrom("Initialize Build Chain", "docker.image.tomorrow")
          textParam("tachyon.app", "tomorrow")
          textParam("tachyon.platform", "mobile_web")

          // this should match the "Tomorrow Production" list to pre-cache these reqs
          snapshotDependency("Lint", "Typecheck")
          snapshotDependency("Test", "Package Tests")
          snapshotDependency("Test", "Tomorrow Tests")
          snapshotDependency("Build", "Build Tomorrow")

          // auto-triggers for main only
          vcsTrigger {
              branchFilter = """
                  +:<default>
              """.trimIndent()
          }

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
          ecsAccountId = "709603242682"
          ecsCluster = "starshot"
          ecsService = "2fsgxn2ycmkf1ilujxl3"

          deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          steps {
              script {
                  name = "Filter Non-Main Deploys"
                  scriptContent = """
                      #!/usr/bin/env bash
                      if [ "%teamcity.build.branch%" != "main" ]; then
                          echo ##teamcity[buildProblem description='Staging slot "2fsgxn2ycmkf1ilujxl3" is reserved for "main" releases only']
                          exit 1
                      fi
                  """.trimIndent()
              }
          }

          updateRegion(AWSRegion.PDX, "tasks/staging.json", 1)
      }

      updateECSService("Starshot Main Staging (old pool)") {
          paramFrom("Initialize Build Chain", "git.commit.short")
          paramFrom("Initialize Build Chain", "docker.image.starshot")
          textParam("tachyon.app", "starshot")
          textParam("tachyon.platform", "staging_web_tv")

          // this should match the "Starshot Production" list to pre-cache these reqs
          snapshotDependency("Lint", "Typecheck")
          snapshotDependency("Test", "Package Tests")
          snapshotDependency("Test", "Starshot Tests")
          snapshotDependency("Build", "Build Starshot")

          // auto-triggers for main only
          vcsTrigger {
              branchFilter = """
                  +:<default>
              """.trimIndent()
          }

          // translates to `docker.image` in task via instrumentorum
          dockerImage = "%ecr.domain%/%docker.image.starshot%"
          ecsAccountId = "709603242682"
          ecsCluster = "starshot"
          ecsService = "87hn6uil7zv5hvmly5mz"

          deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

          failureConditions {
              executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
          }

          steps {
              script {
                  name = "Filter Non-Main Deploys"
                  scriptContent = """
                      #!/usr/bin/env bash
                      if [ "%teamcity.build.branch%" != "main" ]; then
                          echo ##teamcity[buildProblem description='Staging slot "87hn6uil7zv5hvmly5mz" is reserved for "main" releases only']
                          exit 1
                      fi
                  """.trimIndent()
              }
          }

          updateRegion(AWSRegion.PDX, "tasks/staging.json", 1)
      }
    }

    subInstrum("Tomorrow Production", "Tomorrow") {
        dockerAwareBuild("Static Asset Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.tomorrow")

            snapshotDependency("Lint", "Typecheck")
            snapshotDependency("Test", "Package Tests")
            snapshotDependency("Test", "Tomorrow Tests")
            snapshotDependency("Build", "Build Tomorrow")

            steps {
                script {
                    name = "Copy static directories from image"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir -p tmw-static/_next
                        cp -R /opt/tachyon/apps/tomorrow/public/static tmw-static
                        cp -R /opt/tachyon/apps/tomorrow/.next/static tmw-static/_next/
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
                }

                script {
                    name = "Sync static to S3"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        IFS=" " read AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SESSION_TOKEN <<< ${'$'}(aws sts assume-role --role-arn %arn.upload-role% --role-session-name tcagent-tachyon-s3-upload --query 'Credentials.[AccessKeyId,SecretAccessKey,SessionToken]' --output text)
                        export AWS_ACCESS_KEY_ID ; export AWS_SECRET_ACCESS_KEY ; export AWS_SESSION_TOKEN
                        aws s3 sync tmw-static/ "s3://%bucket.js1%" --no-follow-symlinks
                        aws s3 sync tmw-static/ "s3://%bucket.js2%" --no-follow-symlinks
                        aws s3 sync tmw-static/ "s3://%bucket.js3%" --no-follow-symlinks
                        aws s3 sync tmw-static/ "s3://%bucket.js4%" --no-follow-symlinks
                    """.trimIndent()
                }
            }
        }

        buildGroup {
            updateECSService("Canary Deploy") {
                paramFrom("Initialize Build Chain", "git.commit.short")
                paramFrom("Initialize Build Chain", "docker.image.tomorrow")

                snapshotDependency("Tomorrow", "Static Asset Deploy")

                maxRunningBuilds = 1

                // translates to `docker.image` in task via instrumentorum
                dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
                ecsAccountId = "015957721237"
                ecsCluster = "services"
                ecsService = "canary"

                deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

                failureConditions {
                    executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
                }

                steps {
                    script {
                        name = "Filter Non-Main Deploys"
                        scriptContent = """
                            #!/usr/bin/env bash
                            if [ "%teamcity.build.branch%" != "main" ] && [ "%teamcity.build.branch%" != "deploy" ]; then
                                echo ##teamcity[buildProblem description='Tomorrow Canary deploys are reserved for "main" releases only']
                                exit 1
                            fi
                        """.trimIndent()
                    }
                }

                updateRegion(AWSRegion.PDX, "tasks/tomorrow/pdx.json", 3)
                updateRegion(AWSRegion.CLE, "tasks/tomorrow/cle.json", 3)
                updateRegion(AWSRegion.DUB, "tasks/tomorrow/dub.json", 3)
                updateRegion(AWSRegion.FRA, "tasks/tomorrow/fra.json", 3)
                updateRegion(AWSRegion.GRU, "tasks/tomorrow/gru.json", 3)
                updateRegion(AWSRegion.SIN, "tasks/tomorrow/sin.json", 3)
                updateRegion(AWSRegion.SYD, "tasks/tomorrow/syd.json", 3)
            }

            updateECSService("Full Deploy") {
                paramFrom("Initialize Build Chain", "git.commit.short")
                paramFrom("Initialize Build Chain", "docker.image.tomorrow")

                textParam("canary.delay", "30", "Minutes to Delay after Canary Deploy", display = ParameterDisplay.PROMPT)

                snapshotDependency("Tomorrow", "Canary Deploy") {
                    reuseBuilds = ReuseBuilds.NO // always trigger canary before the full production deploy
                }

                maxRunningBuilds = 1

                // translates to `docker.image` in task via instrumentorum
                dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
                ecsAccountId = "015957721237"
                ecsCluster = "services"
                ecsService = "tachyon"

                deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

                steps {
                    script {
                        name = "Filter Non-Main Deploys"
                        scriptContent = """
                            #!/usr/bin/env bash
                            if [ "%teamcity.build.branch%" != "main" ] && [ "%teamcity.build.branch%" != "deploy" ]; then
                                echo ##teamcity[buildProblem description='Tomorrow Production deploys are reserved for "main" releases only']
                                exit 1
                            fi
                        """.trimIndent()
                    }

                    script {
                        name = "Delay after Canary Deploy"
                        scriptContent = """
                            #!/usr/bin/env ruby
                            ${'$'}stdout.sync = true
                            puts "Waiting %canary.delay% minutes before continuing the Production deploy"
                            sleep %canary.delay% * 60
                        """.trimIndent()
                    }

                    script {
                        name = "Tag deploy"
                        scriptContent = """
                            #!/usr/bin/env bash
                            git config user.email "cpe-ops@twitch.tv"
                            git config user.name "CPE TeamCity"
                            git checkout %git.commit.short%
                            TAG=${'$'}(TZ=America/Los_Angeles date '+tmw_deploy_%%Y-%%m-%%d-H%%HM%%M')
                            git tag ${'$'}TAG
                            git push origin ${'$'}TAG
                        """.trimIndent()
                    }
                }

                updateRegion(AWSRegion.PDX, "tasks/tomorrow/pdx.json", 9)
                updateRegion(AWSRegion.CLE, "tasks/tomorrow/cle.json", 9)
                updateRegion(AWSRegion.DUB, "tasks/tomorrow/dub.json", 9)
                updateRegion(AWSRegion.FRA, "tasks/tomorrow/fra.json", 9)
                updateRegion(AWSRegion.GRU, "tasks/tomorrow/gru.json", 9)
                updateRegion(AWSRegion.SIN, "tasks/tomorrow/sin.json", 9)
                updateRegion(AWSRegion.SYD, "tasks/tomorrow/syd.json", 9)
            }
        }

        buildGroup {
            dockerAwareBuild("Pre-production Static Asset Upload") {
                textParam("preprod.s3.bucket", "cldfrt-empcdk-preprod-cloudfrontstaticbucketffd08-91xkdhjxqrah")
                textParam("preprod.s3.upload-role", "arn:aws:iam::079760542461:role/TeamcityToECSIAMRole-USWest2")

                paramFrom("Initialize Build Chain", "git.commit.short")
                paramFrom("Initialize Build Chain", "docker.image.tomorrow")

                snapshotDependency("Lint", "Typecheck")
                snapshotDependency("Test", "Package Tests")
                snapshotDependency("Test", "Tomorrow Tests")
                snapshotDependency("Build", "Build Tomorrow")

                steps {
                    script {
                        name = "Copy static directories from image"
                        scriptContent = """
                            #!/usr/bin/env bash
                            set -e
                            mkdir -p tmw-static/_next
                            cp -R /opt/tachyon/apps/tomorrow/public/static tmw-static
                            cp -R /opt/tachyon/apps/tomorrow/.next/static tmw-static/_next/
                        """.trimIndent()
                        dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
                    }

                    script {
                        name = "Sync static to S3"
                        scriptContent = """
                            #!/usr/bin/env bash
                            set -e
                            IFS=" " read AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SESSION_TOKEN <<< ${'$'}(aws sts assume-role --role-arn %preprod.s3.upload-role% --role-session-name tcagent-tachyon-s3-upload --query 'Credentials.[AccessKeyId,SecretAccessKey,SessionToken]' --output text)
                            export AWS_ACCESS_KEY_ID ; export AWS_SECRET_ACCESS_KEY ; export AWS_SESSION_TOKEN
                            aws s3 sync tmw-static/ "s3://%preprod.s3.bucket%" --no-follow-symlinks
                        """.trimIndent()
                    }
                }
            }

            updateECSService("Pre-production Deploy") {
                textParam("preprod.ecs.service", "B8xzmLHZsXidAyJfNPqK")

                paramFrom("Initialize Build Chain", "git.commit.short")
                paramFrom("Initialize Build Chain", "docker.image.tomorrow")

                snapshotDependency("Tomorrow", "Pre-production Static Asset Upload")

                // translates to `docker.image` in task via instrumentorum
                dockerImage = "%ecr.domain%/%docker.image.tomorrow%"
                ecsAccountId = "079760542461"
                ecsCluster = "EmpEcsServiceCluster"
                ecsService = "%preprod.ecs.service%EcsService"

                deployRole = "TeamcityToECSIAMRole-USWest2"

                failureConditions {
                    executionTimeoutMin = 5 // fail if deploy goes longer than 5 minutes
                }

                steps {
                    script {
                        name = "Filter Non-Main Deploys"
                        scriptContent = """
                            #!/usr/bin/env bash
                            if [ "%teamcity.build.branch%" != "main" ]; then
                                echo ##teamcity[buildProblem description='Tomorrow Pre-production deploys are reserved for "main" releases only']
                                exit 1
                            fi
                        """.trimIndent()
                    }
                }

                updateRegion(AWSRegion.PDX, "tasks/tmw/preprod.json", 1)
            }
        }
    }

    subInstrum("Starshot Production", "Starshot") {
        dockerAwareBuild("Static Asset Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")

            snapshotDependency("Lint", "Typecheck")
            snapshotDependency("Test", "Package Tests")
            snapshotDependency("Test", "Starshot Tests")
            snapshotDependency("Build", "Build Starshot")

            steps {
                script {
                    name = "Copy namespaced static directories from image"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir -p sst-static/_next sst-static/static/starshot
                        cp -R /opt/tachyon/apps/starshot/public/static/starshot sst-static/static/starshot
                        cp -R /opt/tachyon/apps/starshot/.next/static sst-static/_next/
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.starshot%"
                }

                script {
                    name = "Sync static to S3"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        IFS=" " read AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SESSION_TOKEN <<< ${'$'}(aws sts assume-role --role-arn %arn.upload-role% --role-session-name tcagent-tachyon-s3-upload --query 'Credentials.[AccessKeyId,SecretAccessKey,SessionToken]' --output text)
                        export AWS_ACCESS_KEY_ID ; export AWS_SECRET_ACCESS_KEY ; export AWS_SESSION_TOKEN
                        aws s3 sync sst-static/ "s3://%bucket.js1%" --no-follow-symlinks
                        aws s3 sync sst-static/ "s3://%bucket.js2%" --no-follow-symlinks
                        aws s3 sync sst-static/ "s3://%bucket.js3%" --no-follow-symlinks
                        aws s3 sync sst-static/ "s3://%bucket.js4%" --no-follow-symlinks
                    """.trimIndent()
                }
            }
        }

        updateECSService("Web Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")
            textParam("tachyon.platform", "web_tv", "Tachyon Platform", "Name of the platform for this Starshot instance")

            snapshotDependency("Starshot", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.starshot%"
            ecsAccountId = "015957721237"
            ecsCluster = "starshot"
            ecsService = "starshot_7n44uwlq7reik5rwn7ld"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-Main Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "main" ] && [ "%teamcity.build.branch%" != "deploy" ]; then
                            echo ##teamcity[buildProblem description='Starshot Web deploys are reserved for "main" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+sst_web_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/starshot/pdx.json", 6)
            updateRegion(AWSRegion.CLE, "tasks/starshot/cle.json", 6)
            updateRegion(AWSRegion.DUB, "tasks/starshot/dub.json", 6)
            updateRegion(AWSRegion.FRA, "tasks/starshot/fra.json", 6)
            updateRegion(AWSRegion.GRU, "tasks/starshot/gru.json", 6)
            updateRegion(AWSRegion.SIN, "tasks/starshot/sin.json", 6)
            updateRegion(AWSRegion.SYD, "tasks/starshot/syd.json", 6)
        }

        updateECSService("LG QA Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")
            textParam("tachyon.platform", "lg_web_tv", "Tachyon Platform", "Name of the platform for this Starshot instance")

            snapshotDependency("Starshot", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.starshot%"
            ecsAccountId = "015957721237"
            ecsCluster = "starshot"
            ecsService = "starshot_lq8cyqoeb1rtetuux1bx7c"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-LG QA Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "lg_qa" ]; then
                            echo ##teamcity[buildProblem description='LG QA deploys are reserved for "lg" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+sst_lgqa_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/starshot/pdx.json", 6)
            updateRegion(AWSRegion.CLE, "tasks/starshot/cle.json", 6)
            updateRegion(AWSRegion.DUB, "tasks/starshot/dub.json", 6)
            updateRegion(AWSRegion.FRA, "tasks/starshot/fra.json", 6)
            updateRegion(AWSRegion.GRU, "tasks/starshot/gru.json", 6)
            updateRegion(AWSRegion.SIN, "tasks/starshot/sin.json", 6)
            updateRegion(AWSRegion.SYD, "tasks/starshot/syd.json", 6)
        }

        updateECSService("LG Production Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")
            textParam("tachyon.platform", "lg_web_tv", "Tachyon Platform", "Name of the platform for this Starshot instance")

            snapshotDependency("Starshot", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.starshot%"
            ecsAccountId = "015957721237"
            ecsCluster = "starshot"
            ecsService = "starshot_lkzs92r230i7zisfd421y"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-LG Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "lg" ]; then
                            echo ##teamcity[buildProblem description='LG Production deploys are reserved for "lg" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+sst_lg_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/starshot/pdx.json", 6)
            updateRegion(AWSRegion.CLE, "tasks/starshot/cle.json", 6)
            updateRegion(AWSRegion.DUB, "tasks/starshot/dub.json", 6)
            updateRegion(AWSRegion.FRA, "tasks/starshot/fra.json", 6)
            updateRegion(AWSRegion.GRU, "tasks/starshot/gru.json", 6)
            updateRegion(AWSRegion.SIN, "tasks/starshot/sin.json", 6)
            updateRegion(AWSRegion.SYD, "tasks/starshot/syd.json", 6)
        }

        updateECSService("Vestel Prod Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")
            textParam("tachyon.platform", "vestel_web_tv", "Tachyon Platform", "Name of the platform for this Starshot instance")

            snapshotDependency("Starshot", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.starshot%"
            ecsAccountId = "015957721237"
            ecsCluster = "starshot"
            ecsService = "starshot_rq1gqpr82diiadnih9k64m"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-Vestel Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "vestel" ]; then
                            echo ##teamcity[buildProblem description='Vestel deploys are reserved for "vestel" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+sst_vestel_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/starshot/pdx.json", 6)
            updateRegion(AWSRegion.CLE, "tasks/starshot/cle.json", 6)
            updateRegion(AWSRegion.DUB, "tasks/starshot/dub.json", 6)
            updateRegion(AWSRegion.FRA, "tasks/starshot/fra.json", 6)
            updateRegion(AWSRegion.GRU, "tasks/starshot/gru.json", 6)
            updateRegion(AWSRegion.SIN, "tasks/starshot/sin.json", 6)
            updateRegion(AWSRegion.SYD, "tasks/starshot/syd.json", 6)
        }

        updateECSService("Switch Production Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")
            textParam("tachyon.platform", "switch_web_tv", "Tachyon Platform", "Name of the platform for this Starshot instance")

            snapshotDependency("Starshot", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.starshot%"
            ecsAccountId = "015957721237"
            ecsCluster = "starshot"
            ecsService = "starshot_s75i82pylgjlogyhxxm68"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-Switch Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "switch" ]; then
                            echo ##teamcity[buildProblem description='Switch Production deploys are reserved for "switch" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+sst_switch_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/starshot/pdx.json", 6)
            updateRegion(AWSRegion.CLE, "tasks/starshot/cle.json", 6)
            updateRegion(AWSRegion.DUB, "tasks/starshot/dub.json", 6)
            updateRegion(AWSRegion.FRA, "tasks/starshot/fra.json", 6)
            updateRegion(AWSRegion.GRU, "tasks/starshot/gru.json", 6)
            updateRegion(AWSRegion.SIN, "tasks/starshot/sin.json", 6)
            updateRegion(AWSRegion.SYD, "tasks/starshot/syd.json", 6)
        }

        updateECSService("Android Production Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")
            textParam("tachyon.platform", "androidtv_web_tv", "Tachyon Platform", "Name of the platform for this Starshot instance")

            snapshotDependency("Starshot", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.starshot%"
            ecsAccountId = "015957721237"
            ecsCluster = "starshot"
            ecsService = "starshot_sqxl6of3cgxjb1fb7z4kjm"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-Android Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "android" ]; then
                            echo ##teamcity[buildProblem description='Android Production deploys are reserved for "android" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+sst_droid_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/starshot/pdx.json", 6)
            updateRegion(AWSRegion.CLE, "tasks/starshot/cle.json", 6)
            updateRegion(AWSRegion.DUB, "tasks/starshot/dub.json", 6)
            updateRegion(AWSRegion.FRA, "tasks/starshot/fra.json", 6)
            updateRegion(AWSRegion.GRU, "tasks/starshot/gru.json", 6)
            updateRegion(AWSRegion.SIN, "tasks/starshot/sin.json", 6)
            updateRegion(AWSRegion.SYD, "tasks/starshot/syd.json", 6)
        }

        updateECSService("FireTV Production Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.starshot")
            textParam("tachyon.platform", "firetv_web_tv", "Tachyon Platform", "Name of the platform for this Starshot instance")

            snapshotDependency("Starshot", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.starshot%"
            ecsAccountId = "015957721237"
            ecsCluster = "starshot"
            ecsService = "starshot_r4snxjaxyqq2fayko3din"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-FireTV Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "firetv" ]; then
                            echo ##teamcity[buildProblem description='FireTV Production deploys are reserved for "firetv" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+sst_fire_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/starshot/pdx.json", 3)
            updateRegion(AWSRegion.CLE, "tasks/starshot/cle.json", 3)
            updateRegion(AWSRegion.DUB, "tasks/starshot/dub.json", 3)
            updateRegion(AWSRegion.FRA, "tasks/starshot/fra.json", 3)
            updateRegion(AWSRegion.GRU, "tasks/starshot/gru.json", 3)
            updateRegion(AWSRegion.SIN, "tasks/starshot/sin.json", 3)
            updateRegion(AWSRegion.SYD, "tasks/starshot/syd.json", 3)
        }
    }

    subInstrum("Valence Production", "Valence") {
        dockerAwareBuild("Static Asset Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.valence")

            snapshotDependency("Lint", "Typecheck")
            snapshotDependency("Test", "Package Tests")
            snapshotDependency("Test", "Valence Tests")
            snapshotDependency("Build", "Build Valence")

            steps {
                script {
                    name = "Copy static directories from image"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        mkdir -p vlc-static/_next
                        cp -R /opt/tachyon/apps/valence/.next/static vlc-static/_next/
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.valence%"
                }

                script {
                    name = "Sync static to S3"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        IFS=" " read AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SESSION_TOKEN <<< ${'$'}(aws sts assume-role --role-arn %arn.upload-role% --role-session-name tcagent-tachyon-s3-upload --query 'Credentials.[AccessKeyId,SecretAccessKey,SessionToken]' --output text)
                        export AWS_ACCESS_KEY_ID ; export AWS_SECRET_ACCESS_KEY ; export AWS_SESSION_TOKEN
                        aws s3 sync vlc-static/ "s3://%bucket.js1%" --no-follow-symlinks
                        aws s3 sync vlc-static/ "s3://%bucket.js2%" --no-follow-symlinks
                        aws s3 sync vlc-static/ "s3://%bucket.js3%" --no-follow-symlinks
                        aws s3 sync vlc-static/ "s3://%bucket.js4%" --no-follow-symlinks
                    """.trimIndent()
                }
            }
        }

        updateECSService("Production Deploy") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.valence")

            snapshotDependency("Valence", "Static Asset Deploy")

            maxRunningBuilds = 1

            // translates to `docker.image` in task via instrumentorum
            dockerImage = "%ecr.domain%/%docker.image.valence%"
            ecsAccountId = "015957721237"
            ecsCluster = "valence"
            ecsService = "valence_wxytehjpbfegnjhpvtrn"

            deployRole = "TeamcityToECSIAMRole-USWest2-a8c2cb91588648a68930a6cc0e092be018"

            steps {
                script {
                    name = "Filter Non-Valence Deploys"
                    scriptContent = """
                        #!/usr/bin/env bash
                        if [ "%teamcity.build.branch%" != "valence" ]; then
                            echo ##teamcity[buildProblem description='Valence Production deploys are reserved for "valence" releases only']
                            exit 1
                        fi
                    """.trimIndent()
                }

                script {
                    name = "Tag deploy"
                    scriptContent = """
                        #!/usr/bin/env bash
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git checkout %git.commit.short%
                        TAG=${'$'}(TZ=America/Los_Angeles date '+vlc_deploy_%%Y-%%m-%%d-H%%HM%%M')
                        git tag ${'$'}TAG
                        git push origin ${'$'}TAG
                    """.trimIndent()
                }
            }

            updateRegion(AWSRegion.PDX, "tasks/valence/pdx.json", 9)
            updateRegion(AWSRegion.CLE, "tasks/valence/cle.json", 9)
            updateRegion(AWSRegion.FRA, "tasks/valence/fra.json", 9)
            updateRegion(AWSRegion.SIN, "tasks/valence/sin.json", 9)
        }
    }

    subInstrum("Sky Map", "Sky Map") {
        dockerAwareBuild("GH Pages Publish") {
            paramFrom("Initialize Build Chain", "git.commit.short")
            paramFrom("Initialize Build Chain", "docker.image.packages")

            snapshotDependency("Prepare", "Build Packages")
            snapshotDependency("Test", "Sky Map Tests")

            // auto-triggers for main only
            vcsTrigger {
                branchFilter = """
                    +:<default>
                """.trimIndent()
            }

            steps {
                script {
                    name = "Export updated docs"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        export NEXT_DOCS_DEST_DIR=$(pwd)
                        cd /opt/tachyon/apps/sky-map
                        yarn export
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Publish docs to gh-pages branch"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git stash -u
                        git fetch origin gh-pages
                        git checkout gh-pages 2>/dev/null || git checkout -b gh-pages
                        git branch -u origin/gh-pages
                        git reset --hard %git.commit.short%
                        git stash pop
                        git add -A :/
                        git commit -m "Build docs site"
                        git push -f
                    """.trimIndent()
                }
            }
        }
    }

    subInstrum("CI Helpers", "CI") {
        // TODO: convert this into a composite build type
        dockerAwareBuild("Full CI Suite") {
            snapshotDependency("Lint", "Typecheck") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Lint", "Lint TypeScript") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Lint", "Lint Styles") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Lint", "Lint Markdown") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Lint", "Lint package.json") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Lint", "Verify i18n") {
                onDependencyFailure = FailureAction.IGNORE
            }

            snapshotDependency("Test", "Package Tests") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Test", "Laser Array Tests") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Test", "Moonbase Tests") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Test", "Sky Map Tests") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Test", "Starshot Tests") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Test", "Tomorrow Tests") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Test", "Valence Tests") {
                onDependencyFailure = FailureAction.IGNORE
            }

            snapshotDependency("Build", "Build Moonbase") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Build", "Build Tomorrow") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Build", "Build Starshot") {
                onDependencyFailure = FailureAction.IGNORE
            }
            snapshotDependency("Build", "Build Valence") {
                onDependencyFailure = FailureAction.IGNORE
            }

            // auto-triggers for PRs only
            vcsTrigger {
                branchFilter = """
                    +:pull/*
                """.trimIndent()
            }

            steps {
                script {
                    name = "Noop"
                    scriptContent = """
                        #!/usr/bin/env bash
                        echo 'All CI checks run for %teamcity.build.branch% branch.'
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Prettier Markdown") {
            paramFrom("Initialize Build Chain", "docker.image.packages")
            paramFrom("Initialize Build Chain", "git.commit.short")

            snapshotDependency("Prepare", "Build Packages")

            steps {
                script {
                    name = "Run prettier"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        yarn install
                        yarn lint:md:fix
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Push fixes to GHE"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git stash -u
                        git fetch --all
                        git checkout $(git branch -r --contains %git.commit.short% | sed s/origin\\\///)
                        git branch -u $(git branch -r --contains %git.commit.short%)
                        git pull
                        git stash pop
                        git add -A :/
                        git commit -m "Prettier markdown"
                        git push
                    """.trimIndent()
                }
            }
        }

        dockerAwareBuild("Prettier package.json") {
            paramFrom("Initialize Build Chain", "docker.image.packages")
            paramFrom("Initialize Build Chain", "git.commit.short")

            snapshotDependency("Prepare", "Build Packages")

            steps {
                script {
                    name = "Run prettier-package-json"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        yarn install
                        yarn lint:package:fix
                    """.trimIndent()
                    dockerImage = "%ecr.domain%/%docker.image.packages%"
                }

                script {
                    name = "Push fixes to GHE"
                    scriptContent = """
                        #!/usr/bin/env bash
                        set -e
                        git config user.email "cpe-ops@twitch.tv"
                        git config user.name "CPE TeamCity"
                        git stash -u
                        git fetch --all
                        git checkout $(git branch -r --contains %git.commit.short% | sed s/origin\\\///)
                        git branch -u $(git branch -r --contains %git.commit.short%)
                        git pull
                        git stash pop
                        git add -A :/
                        git commit -m "Prettier package.json"
                        git push
                    """.trimIndent()
                }
            }
        }
    }
}
