import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { useCurrentUser } from 'tachyon-auth';
import { useCustomTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AlignItems,
  Avatar,
  CoreInteractive,
  CoreText,
  Display,
  DropDownMenu,
  DropDownMenuItem,
  DropDownMenuSeparator,
  JustifyContent,
  Layout,
  Presence,
  PresenceStatus,
  useDialogState,
} from 'twitch-core-ui';
import type { SpadeClickLogoutButtonEvent } from '../../../../utils';
import { ClickLogoutButtonEvent } from '../../../../utils';
import type { TopNavDropDownMenu_userInfo$key } from './__generated__/TopNavDropDownMenu_userInfo.graphql';

const topNavDropDownMenuFragment = graphql`
  fragment TopNavDropDownMenu_userInfo on UserInfo {
    profileImageURL
    login
    displayName
  }
`;

export type TopNavDropDownMenuProps = {
  userInfo: TopNavDropDownMenu_userInfo$key;
};

export const TopNavDropDownMenu: FC<TopNavDropDownMenuProps> = ({
  userInfo: userInfoRef,
}) => {
  const { displayName, login, profileImageURL } = useFragment(
    topNavDropDownMenuFragment,
    userInfoRef,
  );
  const { anchorProps, dialogProps } = useDialogState();
  const { formatMessage } = useIntl();
  const { logout } = useCurrentUser();
  const trackLogoutEvent = useCustomTracking<SpadeClickLogoutButtonEvent>();
  const avatarAlt = formatMessage('User Avatar', 'MoonbaseUserAvatar');

  return (
    <>
      <CoreInteractive {...anchorProps}>
        <Avatar
          alt={avatarAlt}
          presenceIndicator
          presenceStatus={PresenceStatus.Online}
          size={36}
          src={profileImageURL}
          userLogin={login}
        />
      </CoreInteractive>
      <DropDownMenu {...dialogProps}>
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          justifyContent={JustifyContent.Start}
          padding={{ x: 1, y: 0.5 }}
        >
          <Layout margin={{ right: 0.5 }}>
            <Avatar
              alt={avatarAlt}
              size={36}
              src={profileImageURL}
              userLogin={login}
            />
          </Layout>
          <Layout margin={{ left: 0.5 }}>
            <CoreText bold>
              {getFormattedUserDisplayName({
                // users-service tells us login and displayName are pointers so they
                // may be null but that is for legacy reasons and is extremely unlikely
                displayName: displayName ?? '',
                login: login ?? '',
              })}
            </CoreText>
            <Layout alignItems={AlignItems.Center} display={Display.Flex}>
              <Presence status={PresenceStatus.Online} />
              <Layout margin={{ left: 0.5 }}>
                <CoreText>
                  {formatMessage('Online', 'MoonbaseNavbarStatus')}
                </CoreText>
              </Layout>
            </Layout>
          </Layout>
        </Layout>

        <DropDownMenuSeparator />
        <DropDownMenuItem
          children={formatMessage('Log Out', 'Moonbase')}
          onClick={() => {
            trackLogoutEvent({ event: ClickLogoutButtonEvent.Name });
            logout();
          }}
        />
      </DropDownMenu>
    </>
  );
};

TopNavDropDownMenu.displayName = 'TopNavDropDownMenu';
