import type { FC, FormEvent } from 'react';
import { useEffect } from 'react';
import { graphql, useRefetchableFragment } from 'react-relay/hooks';
import { useCustomTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import type {
  CoreInteractiveElement,
  PaginationMouseEvent,
} from 'twitch-core-ui';
import {
  AlignItems,
  BorderRadius,
  Color,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Pagination,
  Selectable,
  SelectableType,
  WordBreak,
} from 'twitch-core-ui';
import { ENFORCEMENTS_PAGE_SIZE } from '../../../../../config';
import { ViewedEnforcementsPageEvent } from '../../../../../utils';
import type {
  SpadeViewedEnforcementsPageClickEvent,
  SpadeViewedEnforcementsPageClickEventPerEnforcementData,
} from '../../../../../utils';
import { EnforcementTitle } from './EnforcementTitle';
import { StatusPill } from './StatusPill';
import type { EnforcementsSelector_enforcementsInfoForAppeals_enforcements$key } from './__generated__/EnforcementsSelector_enforcementsInfoForAppeals_enforcements.graphql';

const enforcementsInfoForAppealsFragment = graphql`
  fragment EnforcementsSelector_enforcementsInfoForAppeals_enforcements on Query
  @argumentDefinitions(limit: { type: "Int!" }, offset: { type: "Int!" })
  @refetchable(queryName: "enforcementsInfoForAppeals_RefetchQuery") {
    enforcementsInfoForAppeals(pageInfo: { limit: $limit, offset: $offset }) {
      enforcements {
        id
        detailedReason
        appealable
        appealStatus
        medium
        enforcementType
        createdAt
        suspension {
          duration
          isPermanent
        }
      }
      total
    }
  }
`;

export type EnforcementsSelectorProps = {
  currentPage: number;
  enforcementsRef: EnforcementsSelector_enforcementsInfoForAppeals_enforcements$key;
  onChange: (e: FormEvent<HTMLInputElement>, i: number) => void;
  onClickPagination: (pageNum: number) => void;
  selectedEnforcementID: string;
};

export const EnforcementsSelector: FC<EnforcementsSelectorProps> = ({
  currentPage,
  enforcementsRef,
  onChange,
  onClickPagination,
  selectedEnforcementID,
}) => {
  const { formatMessage } = useIntl();
  const trackEvent = useCustomTracking<SpadeViewedEnforcementsPageClickEvent>();
  const [data, refetch] = useRefetchableFragment(
    enforcementsInfoForAppealsFragment,
    enforcementsRef,
  );

  const enforcements = data.enforcementsInfoForAppeals?.enforcements;
  useEffect(() => {
    if (enforcements) {
      const perEnforcementData =
        enforcements.reduce<SpadeViewedEnforcementsPageClickEventPerEnforcementData>(
          (agg, _, i) => {
            agg[`enforcement_id_${i + 1}`] = enforcements[i]?.id ?? '';
            agg[`enforcement_id_${i + 1}_is_appealable`] =
              enforcements[i]?.appealable ?? false;
            return agg;
          },
          {},
        );

      trackEvent({
        ...perEnforcementData,
        event: ViewedEnforcementsPageEvent.Name,
        pageNumber: currentPage,
      });
    }
  }, [enforcements, currentPage, trackEvent]);

  useEffect(() => {
    const offset = (currentPage - 1) * ENFORCEMENTS_PAGE_SIZE;
    refetch({ offset });
  }, [currentPage, refetch]);

  return !enforcements ? null : (
    <>
      <Layout>
        {enforcements.map(
          (e, i) =>
            e && (
              <Layout key={e.id} margin={{ y: 1 }}>
                <Selectable
                  borderRadius={BorderRadius.Large}
                  checked={e.id === selectedEnforcementID}
                  disabled={!e.appealable}
                  name="category-selector-radio"
                  onChange={(evt) => {
                    onChange(evt, i + 1);
                  }}
                  type={SelectableType.Radio}
                  value={e.id}
                >
                  <Layout
                    breakpointExtraSmall={{
                      alignItems: AlignItems.Center,
                      flexDirection: FlexDirection.Row,
                      justifyContent: JustifyContent.Between,
                      padding: { x: 2, y: 1 },
                    }}
                    display={Display.Flex}
                    flexDirection={FlexDirection.Column}
                    padding={{ x: 1, y: 1 }}
                  >
                    <Layout
                      display={Display.Flex}
                      flexDirection={FlexDirection.Column}
                    >
                      <EnforcementTitle
                        suspensionDuration={e.suspension?.duration}
                        type={e.enforcementType}
                      />
                      <Layout padding={{ y: 0.5 }}>
                        <CoreText
                          color={Color.Alt2}
                          wordBreak={WordBreak.BreakWord}
                        >
                          {
                            // NOTE: not in a formatMessage() because enforcement reasons are localized on the backend
                            e.detailedReason
                          }
                        </CoreText>
                      </Layout>
                      <CoreText
                        color={Color.Alt2}
                        wordBreak={WordBreak.BreakWord}
                      >
                        {formatMessage(
                          'received on {date}',
                          {
                            date: new Date(e.createdAt).toLocaleDateString(),
                          },
                          'Moonbase',
                        )}
                      </CoreText>
                    </Layout>
                    <Layout
                      breakpointExtraSmall={{ padding: { y: 1 } }}
                      padding={{ top: 1 }}
                    >
                      <StatusPill appealStatus={e.appealStatus} />
                    </Layout>
                  </Layout>
                </Selectable>
              </Layout>
            ),
        )}
      </Layout>
      <Layout
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        margin={{ top: 2 }}
      >
        <Pagination
          currentPage={currentPage}
          nextPageButtonAriaLabel={formatMessage('Next Page', 'Moonbase')}
          onClickIndex={(e: PaginationMouseEvent<CoreInteractiveElement>) => {
            onClickPagination(e.goToPage);
          }}
          onClickNext={() => {
            onClickPagination(currentPage + 1);
          }}
          onClickPrevious={() => {
            onClickPagination(currentPage - 1);
          }}
          previousPageButtonAriaLabel={formatMessage(
            'Previous Page',
            'Moonbase',
          )}
          totalPages={Math.ceil(
            data.enforcementsInfoForAppeals.total / ENFORCEMENTS_PAGE_SIZE,
          )}
        />
      </Layout>
    </>
  );
};

EnforcementsSelector.displayName = 'EnforcementsSelector';
