import type { FC, ReactChild } from 'react';
import type { TrackEvent } from 'tachyon-event-tracker';
import { useCustomTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  CheckBox,
  CoreLink,
  CoreText,
  Display,
  FormGroup,
  InjectLayout,
  Layout,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import type { SpadeClickedCommunityGuidelinesLinkEvent } from '../../../../../utils';
import { ClickedCommunityGuidelinesLinkEvent } from '../../../../../utils';
import type { Homepage_QueryResponse } from '../../__generated__/Homepage_Query.graphql';

type TermsAndConditionsCheckboxProps = {
  checked?: boolean;
  onChange: () => void;
  selectedEnforcementID?: string;
  userInfo: Homepage_QueryResponse['userInfo'];
};

const tosLink = (
  children: ReactChild,
  trackEvent: TrackEvent<SpadeClickedCommunityGuidelinesLinkEvent>,
  userInfo: Homepage_QueryResponse['userInfo'],
  selectedEnforcementID?: string,
) => {
  const href = 'https://www.twitch.tv/p/en/legal/community-guidelines/';
  return (
    <CoreLink
      children={children}
      linkTo="/deferToRenderLink"
      onClick={() => {
        trackEvent({
          enforcement_id: selectedEnforcementID ?? '',
          event: ClickedCommunityGuidelinesLinkEvent.Name,
          outbound_link: href,
          user_id: userInfo?.id ?? '',
          user_username: userInfo?.login ?? '',
        });
      }}
      renderLink={renderTachyonLink({
        route: RouteName.External,
        routeParams: { href },
      })}
      targetBlank
    />
  );
};

export const TermsAndConditionsCheckbox: FC<TermsAndConditionsCheckboxProps> =
  ({ checked, onChange, selectedEnforcementID, userInfo }) => {
    const { formatMessage } = useIntl();
    const trackEvent =
      useCustomTracking<SpadeClickedCommunityGuidelinesLinkEvent>();

    return (
      <FormGroup
        error={checked !== undefined && !checked}
        errorMessage={formatMessage(
          'You must agree to the community guidelines to continue',
          'Moonbase',
        )}
        label={''}
      >
        <Layout alignItems={AlignItems.Center} display={Display.Flex}>
          <CheckBox
            checked={checked}
            id={'tos-checkbox'}
            label={null}
            onChange={onChange}
            required
          />
          <InjectLayout margin={{ left: 0.5 }}>
            <label htmlFor="tos-checkbox">
              <CoreText>
                {formatMessage(
                  'I have read and acknowledge the <x:link>Twitch Community Guidelines</x:link>',
                  {
                    'x:link': (children) =>
                      tosLink(
                        children,
                        trackEvent,
                        userInfo,
                        selectedEnforcementID,
                      ),
                  },
                  'Moonbase',
                )}
              </CoreText>
            </label>
          </InjectLayout>
        </Layout>
      </FormGroup>
    );
  };

TermsAndConditionsCheckbox.displayName = 'TermsAndConditionsCheckbox';
