import type { FC, FormEvent } from 'react';
import { useState } from 'react';
import { graphql, useMutation } from 'react-relay/hooks';
import { useCustomTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { DangerousServerSuspense, convertToUnsafeID } from 'tachyon-relay';
import { useEffectOnce } from 'tachyon-utils';
import {
  AlignItems,
  Background,
  BorderRadius,
  Button,
  ButtonState,
  Color,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Title,
  TitleSize,
} from 'twitch-core-ui';
import type {
  SpadeSubmitButtonClickEvent,
  SpadeViewedAppealsFormEvent,
} from '../../../../utils';
import { SubmitButtonEvent, ViewedAppealsFormEvent } from '../../../../utils';
import { NAVBAR_HEIGHT } from '../../../common';
import type { Homepage_QueryResponse } from '../__generated__/Homepage_Query.graphql';
import { AppealDescription, MAX_DESCRIPTION_LENGTH } from './AppealDescription';
import { ConfirmationPage } from './ConfirmationPage';
import type { EnforcementsSelectorProps } from './EnforcementsSelector';
import { EnforcementsSelector } from './EnforcementsSelector';
import { EnforcementsSelectorPlaceholder } from './PlaceholderEnforcementsSelector';
import { TermsAndConditionsCheckbox } from './TermsAndConditionsCheckbox';
import { TitleAndContext } from './TitleAndContext';
import { UserAccountDetails } from './UserAccountDetails';
import type { AppealsPortal_Create_Enforcement_Appeal_Mutation } from './__generated__/AppealsPortal_Create_Enforcement_Appeal_Mutation.graphql';

export type AppealsPortalProps = Pick<
  EnforcementsSelectorProps,
  'enforcementsRef'
> & {
  enforcementsInfoForAppeals: Homepage_QueryResponse['enforcementsInfoForAppeals'];
  userInfo: Homepage_QueryResponse['userInfo'];
};

export const AppealsPortal: FC<AppealsPortalProps> = ({
  enforcementsInfoForAppeals,
  enforcementsRef,
  userInfo,
}) => {
  const { formatMessage } = useIntl();
  const [agreedToTermsOfService, setAgreedToTermsOfService] =
    useState<boolean>();
  const [appealDescription, setAppealDescription] = useState<string>();
  const [selectedEnforcementID, setSelectedEnforcementID] =
    useState<string>('');
  const [selectedEnforcementIndexOnPage, setSelectedEnforcementIndexOnPage] =
    useState<number>();
  const [currentPage, setCurrentPage] = useState<number>(1);
  const [showConfirmationModal, setShowConfirmationModal] = useState(false);
  const [error, setError] = useState('');
  const trackSubmitEvent = useCustomTracking<SpadeSubmitButtonClickEvent>();
  const trackViewEvent = useCustomTracking<SpadeViewedAppealsFormEvent>();

  useEffectOnce(() => {
    trackViewEvent({
      event: ViewedAppealsFormEvent.Name,
      total_enforcements: enforcementsInfoForAppeals?.total ?? 0,
      user_id: userInfo?.id ?? '',
      user_username: userInfo?.login ?? '',
    });
  });

  const [commitCreateAppeal, appealBeingCreated] =
    useMutation<AppealsPortal_Create_Enforcement_Appeal_Mutation>(graphql`
      mutation AppealsPortal_Create_Enforcement_Appeal_Mutation(
        $appealDescription: String!
        $enforcementID: ID!
      ) {
        createEnforcementAppeal(
          input: {
            appealDescription: $appealDescription
            enforcementID: $enforcementID
          }
        ) {
          id
          updatedEnforcement {
            id
            appealStatus
            appealable
          }
        }
      }
    `);

  const onSubmit = () => {
    // validate input
    if (!appealDescription) {
      setError(formatMessage('Description can not be empty', 'Moonbase'));
      return;
    } else if (appealDescription.length > MAX_DESCRIPTION_LENGTH) {
      setError(
        formatMessage(
          'Description must be less than {max_characters} characters',
          { max_characters: MAX_DESCRIPTION_LENGTH },
          'Moonbase',
        ),
      );
      return;
    } else if (!selectedEnforcementID) {
      setError(
        formatMessage('You must select an enforcement to continue', 'Moonbase'),
      );
      return;
    }

    const errorCommitMessage = formatMessage(
      'Something went wrong, please try again later',
      'Moonbase',
    );

    commitCreateAppeal({
      onCompleted(resp, err) {
        trackSubmitEvent({
          current_page: currentPage,
          description: appealDescription,
          enforcement_id: convertToUnsafeID(selectedEnforcementID),
          enforcement_index_on_page: selectedEnforcementIndexOnPage ?? 0,
          error: err && err.length > 0 ? err[0].message : '',
          event: SubmitButtonEvent.Name,
          submitted_successfully: err === null,
          total_enforcements: enforcementsInfoForAppeals?.total ?? 0,
          user_id: convertToUnsafeID(userInfo?.id) ?? '',
          user_username: userInfo?.login ?? '',
        });
        if (resp.createEnforcementAppeal?.id) {
          // if appeal was successfully created, we will (1) manually clear selected enforcement
          // because interactables don't do this by default, whereas other form fields do
          // and (2) show the confirmation modal
          setSelectedEnforcementIndexOnPage(undefined);
          setSelectedEnforcementID('');
          setShowConfirmationModal(true);
        } else {
          setError(errorCommitMessage);
        }
        return;
      },
      onError(_) {
        setError(errorCommitMessage);
      },
      variables: {
        appealDescription,
        enforcementID: selectedEnforcementID,
      },
    });
  };

  return showConfirmationModal ? (
    <ConfirmationPage
      onClose={() => {
        setShowConfirmationModal(false);
      }}
    />
  ) : (
    <Layout
      alignItems={AlignItems.Center}
      as="main"
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      justifyContent={JustifyContent.Center}
      minHeight={`calc(100vh - ${NAVBAR_HEIGHT})`}
    >
      <Layout maxWidth="60rem" padding={2}>
        <Layout padding={{ bottom: 1, top: 3 }}>
          <TitleAndContext isSuspended={!!userInfo?.isSuspended} />
        </Layout>
        <Layout
          background={Background.Alt}
          borderRadius={BorderRadius.Large}
          margin={{ top: 3 }}
          padding={3}
        >
          <Title size={TitleSize.Small}>
            {formatMessage(
              'Which enforcement would you like to appeal?',
              'Moonbase',
            )}
          </Title>
          <Layout margin={{ top: 2 }}>
            <DangerousServerSuspense
              fallback={<EnforcementsSelectorPlaceholder />}
            >
              <EnforcementsSelector
                currentPage={currentPage}
                enforcementsRef={enforcementsRef}
                onChange={(e: FormEvent<HTMLInputElement>, i: number) => {
                  setSelectedEnforcementID(e.currentTarget.value);
                  setSelectedEnforcementIndexOnPage(i);
                }}
                onClickPagination={(pageNum) => {
                  setCurrentPage(pageNum);
                }}
                selectedEnforcementID={selectedEnforcementID}
              />
            </DangerousServerSuspense>
          </Layout>
          <UserAccountDetails userInfo={userInfo} />
          <AppealDescription
            onDescriptionChange={setAppealDescription}
            value={appealDescription}
          />
          <TermsAndConditionsCheckbox
            checked={agreedToTermsOfService}
            onChange={() => {
              setAgreedToTermsOfService((prevState) => !prevState);
            }}
            selectedEnforcementID={selectedEnforcementID}
            userInfo={userInfo}
          />
          <Layout margin={{ top: 2 }}>
            <Button
              disabled={
                !agreedToTermsOfService ||
                !appealDescription ||
                appealDescription.length > MAX_DESCRIPTION_LENGTH ||
                !selectedEnforcementID
              }
              onClick={onSubmit}
              state={
                appealBeingCreated ? ButtonState.Loading : ButtonState.Default
              }
            >
              {formatMessage('Submit Appeal', 'Moonbase')}
            </Button>
            <Layout margin={{ top: 1 }}>
              {error && <CoreText color={Color.Error}>{error}</CoreText>}
            </Layout>
          </Layout>
        </Layout>
      </Layout>
    </Layout>
  );
};

AppealsPortal.displayName = 'AppealsPortal';
