import type { ShallowWrapper } from 'enzyme';
import type { Component, FormEvent } from 'react';
import { useMutation } from 'react-relay/hooks';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { Button, ButtonState, CoreText, Title } from 'twitch-core-ui';
import { homepageQuery } from '../test-mocks';
import { AppealDescription } from './AppealDescription';
import { ConfirmationPage } from './ConfirmationPage';
import { EnforcementsSelector } from './EnforcementsSelector';
import { TermsAndConditionsCheckbox } from './TermsAndConditionsCheckbox';
import { TitleAndContext } from './TitleAndContext';
import { UserAccountDetails } from './UserAccountDetails';
import type { AppealsPortalProps } from '.';
import { AppealsPortal } from '.';

const mockCommit = jest.fn();
jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useMutation: jest.fn(() => [mockCommit, false]),
}));
const mockUseMutation = useMutation as jest.Mock;

const fillOutForm = (
  wrapper: ShallowWrapper<AppealsPortalProps, {}, Component<{}, {}, any>>,
) => {
  // fill out description
  wrapper
    .find(AppealDescription)
    .props()
    .onDescriptionChange('a valid description');
  // check terms and conditions checkbox
  wrapper.find(TermsAndConditionsCheckbox).props().onChange();
  expect(wrapper.find(TermsAndConditionsCheckbox).props().checked).toBe(true);
  // select an enforcement
  wrapper
    .find(EnforcementsSelector)
    .props()
    .onChange(
      {
        currentTarget: { value: 'enforcementID' },
      } as FormEvent<HTMLInputElement>,
      0,
    );

  return wrapper;
};

describe(AppealsPortal, () => {
  const setup = createShallowWrapperFactory(AppealsPortal, () => ({
    enforcementsRef: homepageQuery,
    ...homepageQuery,
  }));

  it('renders as expected', () => {
    const { wrapper } = setup();
    expect(wrapper.find(TitleAndContext)).toExist();
    expect(wrapper.find(Title)).toExist();
    expect(wrapper.find(Title).html()).toContain(
      'Which enforcement would you like to appeal?',
    );
    expect(wrapper.find(EnforcementsSelector)).toExist();
    expect(wrapper.find(UserAccountDetails)).toExist();
    expect(wrapper.find(AppealDescription)).toExist();
    expect(wrapper.find(TermsAndConditionsCheckbox)).toExist();
    expect(wrapper.find(Button)).toExist();
    expect(wrapper.find(Button).text()).toEqual('Submit Appeal');
  });

  it('button is disabled by default and until all fields are complete', () => {
    const { wrapper } = setup();
    // button starts out disabled
    expect(wrapper.find(Button)).toBeDisabled();
    // fill out form
    const filledOutFormWrapper = fillOutForm(wrapper);
    // finally, the button should be clickable
    expect(filledOutFormWrapper.find(Button)).not.toBeDisabled();
  });

  it('button will go disabled again if input becomes invalid again', () => {
    const { wrapper } = setup();
    // button starts out disabled
    expect(wrapper.find(Button)).toBeDisabled();
    // fill out form
    const filledOutFormWrapper = fillOutForm(wrapper);
    expect(filledOutFormWrapper.find(Button)).not.toBeDisabled();
    // invalid description
    filledOutFormWrapper.find(AppealDescription).props().onDescriptionChange(
      // 5001 characters (too long)
      'Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum. Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem. Maecenas nec odio et ante tincidunt tempus. Donec vitae sapien ut libero venenatis faucibus. Nullam quis ante. Etiam sit amet orci eget eros faucibus tincidunt. Duis leo. Sed fringilla mauris sit amet nibh. Donec sodales sagittis magna. Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero. Fusce vulputate eleifend sapien. Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus. Nullam accumsan lorem in dui. Cras ultricies mi eu turpis hendrerit fringilla. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In ac dui quis mi consectetuer lacinia. Nam pretium turpis et arcu. Duis arcu tortor, suscipit eget, imperdiet nec, imperdiet iaculis, ipsum. Sed aliquam ultrices mauris. Integer ante arcu, accumsan a, consectetuer eget, posuere ut, mauris. Praesent adipiscing. Phasellus ullamcorper ipsum rutrum nunc. Nunc nonummy metus. Vestibulum volutpat pretium libero. Cras id dui. Aenean ut eros et nisl sagittis vestibulum. Nullam nulla eros, ultricies sit amet, nonummy id, imperdiet feugiat, pede. Sed lectus. Donec mollis hendrerit risus. Phasellus nec sem in justo pellentesque facilisis. Etiam imperdiet imperdiet orci. Nunc nec neque. Phasellus leo dolor, tempus non, auctor et, hendrerit quis, nisi. Curabitur ligula sapien, tincidunt non, euismod vitae, posuere imperdiet, leo. Maecenas malesuada. Praesent congue erat at massa. Sed cursus turpis vitae tortor. Donec posuere vulputate arcu. Phasellus accumsan cursus velit. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed aliquam, nisi quis porttitor congue, elit erat euismod orci, ac placerat dolor lectus quis orci. Phasellus consectetuer vestibulum elit. Aenean tellus metus, bibendum sed, posuere ac, mattis non, nunc. Vestibulum fringilla pede sit amet augue. In turpis. Pellentesque posuere. Praesent turpis. Aenean posuere, tortor sed cursus feugiat, nunc augue blandit nunc, eu sollicitudin urna dolor sagittis lacus. Donec elit libero, sodales nec, volutpat a, suscipit non, turpis. Nullam sagittis. Suspendisse pulvinar, augue ac venenatis condimentum, sem libero volutpat nibh, nec pellentesque velit pede quis nunc. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce id purus. Ut varius tincidunt libero. Phasellus dolor. Maecenas vestibulum mollis diam. Pellentesque ut neque. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. In dui magna, posuere eget, vestibulum et, tempor auctor, justo. In ac felis quis tortor malesuada pretium. Pellentesque auctor neque nec urna. Proin sapien ipsum, porta a, auctor quis, euismod ut, mi. Aenean viverra rhoncus pede. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Ut non enim eleifend felis pretium feugiat. Vivamus quis mi. Phasellus a est. Phasellus magna. In hac habitasse platea dictumst. Curabitur at lacus ac velit ornare lobortis. Curabitur a felis in nunc fringilla tristique. Morbi mattis ullamcorper velit. Phasellus gravida semper nisi. Nullam vel sem. Pellentesque libero tortor, tincidunt et, tincidunt eget, semper nec, quam. Sed hendrerit. Morbi ac felis. Nunc egestas, augue at pellentesque laoreet, felis eros vehicula leo, at malesuada velit leo quis pede. Donec interdum, metus et hendrerit aliquet, dolor diam sagittis ligula, eget egestas libero turpis vel mi. Nunc nulla. Fusce risus nisl, viverra et, tempor et, pretium in, sapien. Donec venenatis vulputate lorem. Morbi nec metus. Phasellus blandit leo ut odio. Maecenas ullamcorper, dui et placerat feugiat, eros pede varius nisi, condimentum viverra felis nunc et lorem. Sed magna purus, fermentum eu, tincidunt eu, varius ut, felis. In auctor lobortis lacus. Quisque libero metus, condimentum nec, tempor a, commodo mollis, magna. Vestibulum ullamcorper mauris at ligula',
    );
    // the button should be disabled now
    expect(filledOutFormWrapper.find(Button)).toBeDisabled();
  });

  it('clicking submit button puts it in a loading state', () => {
    mockUseMutation.mockImplementation(() => [mockCommit, true]);
    const { wrapper } = setup();
    // fill out form
    const filledOutFormWrapper = fillOutForm(wrapper);
    // click button
    filledOutFormWrapper.find(Button).simulate('click');
    // expect button to be in loading state
    expect(filledOutFormWrapper.find(Button).html()).toContain(
      ButtonState.Loading,
    );
  });

  it('clicking submit button fires mutation and then shows confirmation modal', () => {
    mockCommit.mockImplementationOnce(({ onCompleted }) => {
      onCompleted({ createEnforcementAppeal: { id: 10 } });
    });
    const { wrapper } = setup();
    // confirmation modal does not show to start with
    expect(wrapper.find(ConfirmationPage)).not.toExist();
    // fill out form
    const filledOutFormWrapper = fillOutForm(wrapper);
    // click button
    filledOutFormWrapper.find(Button).simulate('click');
    // expect commit to have been called
    expect(mockCommit).toHaveBeenCalledTimes(1);
    // expect confirmation modal to be shown
    filledOutFormWrapper.update();
    expect(filledOutFormWrapper.find(ConfirmationPage)).toExist();
  });

  it('test that on inadequate payload, we show an error message', () => {
    mockCommit.mockImplementationOnce(({ onCompleted }) => {
      onCompleted({});
    });
    const { wrapper } = setup();
    // fill out form
    const filledOutFormWrapper = fillOutForm(wrapper);
    // click button
    filledOutFormWrapper.find(Button).simulate('click');
    // expect commit to have been called
    expect(mockCommit).toHaveBeenCalledTimes(1);
    // expect error message to be shown
    expect(filledOutFormWrapper.find(CoreText)).toExist();
    expect(filledOutFormWrapper.find(CoreText).text()).toContain(
      'Something went wrong, please try again later',
    );
  });

  it('test that on mutation error, we show an error message', () => {
    mockCommit.mockImplementationOnce(({ onError }) => {
      onError();
    });
    const { wrapper } = setup();
    // fill out form
    const filledOutFormWrapper = fillOutForm(wrapper);
    // click button
    filledOutFormWrapper.find(Button).simulate('click');
    // expect commit to have been called
    expect(mockCommit).toHaveBeenCalledTimes(1);
    // expect error message to be shown
    expect(filledOutFormWrapper.find(CoreText)).toExist();
    expect(filledOutFormWrapper.find(CoreText).text()).toContain(
      'Something went wrong, please try again later',
    );
  });

  it("test that on selecting an enforcement, we update the component from form's state", () => {
    const { wrapper } = setup();
    // button starts out disabled
    expect(wrapper.find(Button)).toBeDisabled();
    // select an enforcement
    wrapper
      .find(EnforcementsSelector)
      .props()
      .onChange(
        {
          currentTarget: { value: 'enforcementID' },
        } as FormEvent<HTMLInputElement>,
        2,
      );

    expect(
      wrapper.find(EnforcementsSelector).props().selectedEnforcementID,
    ).toEqual('enforcementID');
  });
});
