import { graphql } from 'react-relay/hooks';
import { useStaticEnvironment } from 'tachyon-environment';
import type { TachyonPage } from 'tachyon-next-types';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { getCookieValue, setCookieValue, useEffectOnce } from 'tachyon-utils';
import {
  ENFORCEMENTS_PAGE_SIZE,
  USER_LANGUAGE_COOKIE_NAME,
} from '../../../config';
import { LoginGate, TopNav } from '../../common';
import { AppealsPortal } from './AppealsPortal';
import { NoEnforcementsLandingPage } from './NoEnforcementsLandingPage';
import type { Homepage_QueryResponse } from './__generated__/Homepage_Query.graphql';

export type HomepageProps = Homepage_QueryResponse;

export const Homepage: TachyonPage<{}, HomepageProps> = ({
  enforcementsInfoForAppeals,
  userInfo,
  ...props
}) => {
  const {
    common: { language },
  } = useStaticEnvironment();
  const userLang = userInfo?.language;

  useEffectOnce(() => {
    // only reload if:
    // - user has truthy language setting
    // - user language doesn't match currently selected language
    // - user language isn't already set as the cookie value (prevent infinite loop)
    // istanbul ignore next: tested in browser
    if (
      !!userLang &&
      userLang.toLowerCase() !== language.toLowerCase() &&
      userLang !== getCookieValue({ name: USER_LANGUAGE_COOKIE_NAME })
    ) {
      setCookieValue({
        name: USER_LANGUAGE_COOKIE_NAME,
        opts: {
          // If set to undefined, the cookie becomes a session cookie.
          // A session finishes when the client shuts down, and session cookies will be removed.
          // We do this so that user's language cookie gets cleared and doesn't stick around unnecessarily
          expires: undefined,
        },
        value: userLang,
      });
      // refresh the page so new cookie is being used
      location.reload();
    }
  });

  const userHasEnforcements =
    enforcementsInfoForAppeals && enforcementsInfoForAppeals.total > 0;
  return (
    <LoginGate>
      <TopNav userInfo={userInfo} />
      {userHasEnforcements ? (
        <AppealsPortal
          enforcementsInfoForAppeals={enforcementsInfoForAppeals}
          enforcementsRef={props}
          userInfo={userInfo}
        />
      ) : (
        <NoEnforcementsLandingPage userInfo={userInfo} />
      )}
    </LoginGate>
  );
};

Homepage.displayName = 'Homepage';
Homepage.pageviewTracking = defaultPageviewTracking;
Homepage.query = graphql`
  query Homepage_Query($limit: Int!, $offset: Int!) {
    userInfo {
      id
      login
      displayName
      phoneNumber
      email
      emailVerified
      profileImageURL
      isSuspended
      language
      ...TopNavDropDownMenu_userInfo
    }
    ...EnforcementsSelector_enforcementsInfoForAppeals_enforcements
      @arguments(limit: $limit, offset: $offset)
    enforcementsInfoForAppeals(pageInfo: { limit: $limit, offset: $offset }) {
      total
    }
  }
`;
Homepage.getInitialProps = () => ({
  queryVariables: {
    limit: ENFORCEMENTS_PAGE_SIZE,
    offset: 0,
  },
});
