import { useEffect } from 'react';
import { useCurrentUser } from 'tachyon-auth';
import { useDynamicSettings } from 'tachyon-dynamic-settings';
import { useIntl } from 'tachyon-intl';
import { GlowGearSvg } from 'tachyon-more-ui';
import type { TachyonPage } from 'tachyon-next-types';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { isBrowser, useConst } from 'tachyon-utils';
import {
  AlignItems,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  TextAlign,
  Title,
} from 'twitch-core-ui';
import type { MoonbaseDynamicSettings } from '../../../config';
import { RouteName, useTachyonRouter } from '../../../routing';
import { LoginButton, NAVBAR_HEIGHT, TopNav } from '../../common';

function useLoginRedirect(): boolean {
  // Capture the href on first render going down the tree so we don't lose it in
  // a useEffect race as part of URL cleanup (necessary because router-utils
  // doesn't capture fragments
  const href = useConst(() => (isBrowser() ? window.location.href : ''));
  const router = useTachyonRouter();
  const { loggedIn, login } = useCurrentUser();

  useEffect(() => {
    if (!loggedIn) {
      const myUrl = new URL(href.replace(/#/g, '?'));
      const access_token = myUrl.searchParams.get('access_token');

      if (access_token) {
        login({
          access_token,
          refresh_token: '',
        });
      }
    }

    if (loggedIn) {
      router.push({
        route: RouteName.Homepage,
      });
    }
  }, [router, loggedIn, login, href]);

  return loggedIn;
}

export const OAuth: TachyonPage = () => {
  const { formatMessage } = useIntl();
  const redirecting = useLoginRedirect();
  const { moonbaseDisabled } = useDynamicSettings<MoonbaseDynamicSettings>();

  if (moonbaseDisabled) {
    return null;
  }

  if (redirecting) {
    return null;
  }

  return (
    <>
      <TopNav />
      <Layout
        alignItems={AlignItems.Center}
        as="main"
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        justifyContent={JustifyContent.Center}
        minHeight={`calc(100vh - ${NAVBAR_HEIGHT})`}
        padding={2}
      >
        <Layout maxWidth="15rem">
          <GlowGearSvg />
        </Layout>
        <Layout margin={{ y: 1 }}>
          <Title>{formatMessage('Appeals Portal', 'Moonbase')}</Title>
        </Layout>

        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          justifyContent={JustifyContent.Center}
        >
          <Layout margin={{ bottom: 2 }} textAlign={TextAlign.Center}>
            <CoreText>
              {formatMessage(
                'To view your enforcements and submit an appeal, please log in.',
                'Moonbase',
              )}
            </CoreText>
          </Layout>
          <LoginButton />
        </Layout>
      </Layout>
    </>
  );
};

OAuth.displayName = 'OAuth';
OAuth.pageviewTracking = defaultPageviewTracking;
