import type { AppEnvironment } from 'tachyon-environment';
import { configureClientLogging, logger } from 'tachyon-logger';
import type { ConfigureTachyonRelayOpts } from 'tachyon-relay';
import { configureTachyonRelay } from 'tachyon-relay';
import {
  getCurrentTwitchDomain,
  isBrowser,
  setCookieDomain,
} from 'tachyon-utils';
import { CLIENT_ID, DEFAULT_PLATFORM, GQL_ENDPOINT } from '../../../../config';

// istanbul ignore next: mostly testing mocks
export function configure(): void {
  let gqlEndpoint: string = GQL_ENDPOINT['production'];
  if (isBrowser()) {
    const buildId = process.env.BUILD_ID;
    // reach into next data since we're outside of the react tree on the client
    const { appEnvironment, language } = window.__NEXT_DATA__.props;

    gqlEndpoint = GQL_ENDPOINT[appEnvironment as AppEnvironment];

    logger.debug({
      category: 'configure',
      context: {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        appEnvironment,
        buildId,
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        language,
      },
      message: 'Configuring application',
      package: 'moonbase',
    });

    if (process.env.NODE_ENV === 'production') {
      // configure error reporting (this should happen first/asap)
      try {
        configureClientLogging({
          // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
          appEnvironment,
          appSentryDsn: process.env.SENTRY_DSN,
          buildId,
          // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
          language,
          platform: DEFAULT_PLATFORM,
        });
      } catch {
        // If this fails still try to start the app, don't bother logging though since our
        // logger isn't shipping logs.
      }
    }

    // configure cookie domain
    try {
      const domain = getCurrentTwitchDomain(window.location.hostname);
      if (domain) {
        setCookieDomain(domain);
        // TODO: audit necessity
        // https://jira.twitch.com/browse/MWC-2473
        // We need this for certain player cross-frame communication.
        // It makes sense to put this here instead of the player component
        // since it is a fairly global change.
        document.domain = domain;
        logger.debug({
          category: 'configure',
          message: `Setting the domain for cookies and the document to ${domain}`,
          package: 'moonbase',
        });
      }
    } catch {
      // This can throw a security exception if the page isn't on a twitch.tv
      // subdomain. If it does then there isn't really much we can do about
      // this. If we didn't have staging domains off of twitch.tv then we
      // could consider this an _actual_ security error.
      logger.warn({
        category: 'configure',
        message: `Failed to set domain for ${window.location.hostname}`,
        package: 'moonbase',
      });
    }
  }

  const relayOpts: ConfigureTachyonRelayOpts = {
    clientId: CLIENT_ID,
    errorsArrayIsFatal: () => false,
    gqlEndpoint,
  };

  // not using isBrowser() to ensure DCE and thus no `require('https')` in the browser
  if (typeof window === 'undefined') {
    // eslint-disable-next-line @typescript-eslint/no-var-requires, @typescript-eslint/no-unsafe-assignment
    const { Agent } = require('https');
    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-call
    relayOpts.serverHttpsAgent = new Agent({
      keepAlive: true,
    });
    // eslint-disable-next-line @typescript-eslint/no-var-requires, @typescript-eslint/no-unsafe-assignment
    const { getTachyonEnvVar } = require('tachyon-server-utils');
    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-call
    const appEnvironment = getTachyonEnvVar();
    gqlEndpoint = GQL_ENDPOINT[appEnvironment as AppEnvironment];
    relayOpts.gqlEndpoint = gqlEndpoint;
  }

  configureTachyonRelay(relayOpts);
}
