import { getCurrentUserOnServer, logoutOnServer } from 'tachyon-auth-server';
import type { InitEnvironmentOpts, TwitchPayloadError } from 'tachyon-relay';
import { initEnvironment } from 'tachyon-relay';
import { HTTPStatusCode } from 'tachyon-type-library';
import { getServersideInitialProps } from '.';

jest.mock('tachyon-logger', () => ({ logger: { info: jest.fn() } }));

let mockFetchQueryValue: Promise<any> | undefined = undefined;
jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  fetchQuery: jest.fn(() => ({ toPromise: () => mockFetchQueryValue })),
}));

let mockNonFatalErrors = false;
jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  initEnvironment: jest.fn((opts: InitEnvironmentOpts) => {
    if (mockNonFatalErrors) {
      opts.fetchQueryOpts!.nonFatalErrorsHandler!([
        'uh oh',
      ] as unknown as TwitchPayloadError[]);
    }
    return jest.requireActual('tachyon-relay').initEnvironment(opts);
  }),
}));
const mockInitEnvironment = initEnvironment as jest.Mock;

jest.mock('tachyon-auth-server', () => ({
  getCurrentUserOnServer: jest.fn(() => ({ isLoggedIn: false })),
  logoutOnServer: jest.fn(),
}));
const mockLogoutOnServer = logoutOnServer as jest.Mock;
const mockGetCurrentUserOnServer = getCurrentUserOnServer as jest.Mock;

const mockTachyonRes = { setHeader: jest.fn() };

beforeEach(() => {
  mockFetchQueryValue = undefined;
  mockTachyonRes.setHeader.mockReset();
});

describe(getServersideInitialProps, () => {
  const req = { tachyon: { deviceId: 'device id' } };
  describe('authentication', () => {
    it('provides the authToken when present', async () => {
      const mockToken = 'token';
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        authorizationToken: mockToken,
        isLoggedIn: true,
      }));

      await getServersideInitialProps({
        Component: { query: true } as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          req,
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(mockInitEnvironment).toHaveBeenCalledWith(
        expect.objectContaining({
          fetchQueryOpts: expect.objectContaining({
            authorization: expect.objectContaining({ token: mockToken }),
          }),
        }),
      );
    });

    it('logs the user out when unauthorized', async () => {
      const mockToken = 'token';
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        authorizationToken: mockToken,
        isLoggedIn: true,
      }));
      mockInitEnvironment.mockImplementationOnce((opts) => {
        opts.fetchQueryOpts.authorization.unauthorizedHandler();
      });

      const { isLoggedInServerside } = await getServersideInitialProps({
        Component: { query: true } as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          req,
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(mockLogoutOnServer).toHaveBeenCalledTimes(1);
      expect(isLoggedInServerside).toEqual(false);
    });

    it('returns currentUser status', async () => {
      mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
        isLoggedIn: true,
      }));

      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          req,
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.isLoggedInServerside).toEqual(true);
    });
  });

  describe('serversideError', () => {
    it('true when tachyonCtx.err', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          err: new Error('oh no'),
          req,
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.serversideError).toEqual(true);
    });

    it('true when 500 status code', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          req,
          res: { ...mockTachyonRes, statusCode: 500 },
        } as any,
      });

      expect(res.serversideError).toEqual(true);
    });

    it('false otherwise', async () => {
      const res = await getServersideInitialProps({
        Component: {} as any,
        fetchQueryOpts: {} as any,
        relayQueryVariables: {} as any,
        tachyonCtx: {
          req,
          res: { ...mockTachyonRes, statusCode: 200 },
        } as any,
      });

      expect(res.serversideError).toEqual(false);
    });
  });

  describe('relayQueryRecords', () => {
    describe('Component with query', () => {
      it('returns relay query records', async () => {
        mockFetchQueryValue = Promise.resolve(true);

        const res = await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            req,
            res: { ...mockTachyonRes, statusCode: 200 },
          } as any,
        });

        expect(res.relayQueryRecords).not.toBeUndefined();
      });
    });

    describe('Component without query', () => {
      it('does not return relay query records', async () => {
        const res = await getServersideInitialProps({
          Component: {} as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: {
            req,
            res: { ...mockTachyonRes, statusCode: 200 },
          } as any,
        });

        expect(res.relayQueryRecords).toBeUndefined();
      });
    });
  });

  describe('headers', () => {
    describe('Component with query', () => {
      it('request failures set the status code to BadGateway only when user is logged in', async () => {
        const mockToken = 'token';
        mockGetCurrentUserOnServer.mockImplementationOnce(() => ({
          authorizationToken: mockToken,
          isLoggedIn: true,
        }));

        const error = new Error('oh no');
        mockFetchQueryValue = Promise.reject(error);
        const res = { ...mockTachyonRes, statusCode: undefined };

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { req, res } as any,
        });

        expect(res.statusCode).toEqual(HTTPStatusCode.BadGateway);
      });

      it('partial request failures set the status code to BadGateway', async () => {
        mockNonFatalErrors = true;
        const res = { ...mockTachyonRes, statusCode: undefined };

        await getServersideInitialProps({
          Component: { query: true } as any,
          fetchQueryOpts: {} as any,
          relayQueryVariables: {} as any,
          tachyonCtx: { req, res } as any,
        });

        expect(res.statusCode).toEqual(HTTPStatusCode.BadGateway);
        mockNonFatalErrors = false;
      });

      describe('Component.isNotFoundServerside', () => {
        it('sets the status code to NotFound', async () => {
          mockFetchQueryValue = Promise.resolve(true);
          const res = { ...mockTachyonRes, statusCode: undefined };

          await getServersideInitialProps({
            Component: { isNotFoundServerside: () => true, query: true } as any,
            fetchQueryOpts: {} as any,
            relayQueryVariables: {} as any,
            tachyonCtx: { req, res } as any,
          });

          expect(res.statusCode).toEqual(HTTPStatusCode.NotFound);
        });
      });
    });
  });
});
