import App from 'next/app';
import type { FC } from 'react';
import { useState } from 'react';
import type { Variables as RelayQueryVariables } from 'react-relay/hooks';
import { CurrentUserRoot } from 'tachyon-auth';
import type { DynamicSettings } from 'tachyon-dynamic-settings';
import { DynamicSettingsRoot } from 'tachyon-dynamic-settings';
import type { AppEnvironment } from 'tachyon-environment';
import { TachyonIntlRoot, useIntl } from 'tachyon-intl';
import type { ClientSideErrorBoundaryProps } from 'tachyon-more-ui';
import { ClientSideErrorBoundary, CoreUiSsrRoot } from 'tachyon-more-ui';
import { RouterUtilsRoot } from 'tachyon-next-routing-utils';
import type {
  TachyonAppBaseInitialProps,
  TachyonAppBaseProps,
  TachyonAppFC,
  TachyonPageInitialProps,
} from 'tachyon-next-types';
import type { FetchQueryOpts } from 'tachyon-relay';
import { isBrowser } from 'tachyon-utils';
import {
  AlignItems,
  Display,
  JustifyContent,
  Layout,
  Position,
} from 'twitch-core-ui';
import type {
  MoonbaseDynamicSettings,
  MoonbaseRequestExtensions,
} from '../../../config';
import { Error } from '../../common';
import { PageRenderer } from './PageRenderer';
import { configure } from './configure';
import {
  MoonbaseEnvironmentRoot,
  MoonbaseEventReporterRoot,
  MoonbaseEventTrackerRoot,
  MoonbaseExperimentsRoot,
  MoonbaseLatencyTrackerRoot,
  MoonbaseRelayEnvironmentRoot,
} from './context-root-adapters';
import type { MoonbaseServersideInitialProps } from './getServersideInitialProps';
import {
  getServersideInitialProps,
  isServersideContext,
} from './getServersideInitialProps';

configure();

// list of query params to protect from being stripped by the query-param
// handling in RouterUtilsRoot
const PRESERVED_QUERY_PARAMS: string[] = [];

// localized error UI
const ErrorPage: FC = () => {
  const { formatMessage } = useIntl();
  return (
    <Layout
      alignItems={AlignItems.Center}
      attachBottom
      attachLeft
      attachRight
      attachTop
      display={Display.Flex}
      justifyContent={JustifyContent.Center}
      position={Position.Absolute}
    >
      <Error
        description={formatMessage(
          'An unexpected error occurred, please refresh and try again.',
          'Moonbase',
        )}
        title={formatMessage('Oops!', 'Moonbase')}
      />
    </Layout>
  );
};
ErrorPage.displayName = 'ErrorPage';

// non-localized error UI of last resort
const RootErrorMessage: FC = () => (
  <span>An error occurred. Please refresh the page.</span>
);
RootErrorMessage.displayName = 'RootErrorMessage';

/**
 * PersistedMoonbaseAppInitialProps is comprised of values that we need to transport
 * from the server to client, and we lean on next's `getInitialProps`
 * serialization functionality to achieve that. The values are set on the
 * server, consumed from __NEXT_DATA__ via normal next functionality for the
 * first page render, and then manually re-consumed on subsequent app re-renders
 * to ensure their continued availability.
 */
type MoonbasePersistedAppInitialProps = Pick<
  MoonbaseRequestExtensions,
  'experimentBucket' | 'intlData'
> & {
  appEnvironment: AppEnvironment;
  dynamicSettings: DynamicSettings<MoonbaseDynamicSettings>;
  language: string;
};

// Be very conservative about what is put in here, since all of this will get
// serialized for being sent to the client and big/complex objects both don't
// do very well with this and also cause a ton of page bloat.

type MoonbaseAppInitialProps = MoonbasePersistedAppInitialProps &
  Partial<MoonbaseServersideInitialProps> &
  TachyonAppBaseInitialProps & {
    currentAsPath: string;
    fetchQueryOpts: FetchQueryOpts;
    relayQueryVariables: RelayQueryVariables;
  };

export type MoonbaseAppProps = MoonbaseAppInitialProps & TachyonAppBaseProps;

export const MoonbaseApp: TachyonAppFC<
  MoonbaseAppInitialProps,
  MoonbaseAppProps,
  MoonbaseRequestExtensions
> = ({
  Component,
  appEnvironment,
  currentAsPath,
  dynamicSettings,
  experimentBucket,
  fetchQueryOpts,
  intlData,
  isLoggedInServerside,
  language,
  pageProps,
  relayQueryRecords,
  relayQueryVariables,
  serversideError,
}) => {
  const [isInError, setIsInError] = useState(false);
  function onError() {
    setIsInError(true);
  }

  const commonErrorBoundaryProps: Pick<
    ClientSideErrorBoundaryProps,
    'app' | 'currentPath' | 'isDevelopment'
  > = {
    app: 'moonbase',
    currentPath: currentAsPath,
    isDevelopment: process.env.NODE_ENV === 'development',
  };

  return (
    <ClientSideErrorBoundary
      {...commonErrorBoundaryProps}
      boundaryName="MoonbaseRootBoundary"
      errorStateComponent={RootErrorMessage}
      isRootBoundary
    >
      <RouterUtilsRoot preservedParams={PRESERVED_QUERY_PARAMS}>
        <DynamicSettingsRoot settings={dynamicSettings}>
          <MoonbaseEnvironmentRoot
            appEnvironment={appEnvironment}
            language={language}
          >
            <MoonbaseEventReporterRoot>
              <TachyonIntlRoot data={intlData}>
                <CurrentUserRoot isLoggedInServerside={isLoggedInServerside}>
                  <MoonbaseEventTrackerRoot
                    inError={serversideError ?? isInError}
                  >
                    <MoonbaseRelayEnvironmentRoot
                      fetchQueryOpts={fetchQueryOpts}
                      records={relayQueryRecords}
                    >
                      <MoonbaseLatencyTrackerRoot>
                        <MoonbaseExperimentsRoot
                          bucket={experimentBucket}
                          debug={appEnvironment === 'development'}
                          experiments={dynamicSettings.experiments}
                        >
                          <ClientSideErrorBoundary
                            {...commonErrorBoundaryProps}
                            boundaryName="MoonbaseAppBoundary"
                            errorStateComponent={ErrorPage}
                            onError={onError}
                          >
                            <CoreUiSsrRoot
                              appRootElementId="__next"
                              theme="dark"
                            >
                              {serversideError ? (
                                <ErrorPage />
                              ) : (
                                <PageRenderer
                                  Page={Component}
                                  pageProps={pageProps}
                                  relayQueryVariables={relayQueryVariables}
                                />
                              )}
                            </CoreUiSsrRoot>
                          </ClientSideErrorBoundary>
                        </MoonbaseExperimentsRoot>
                      </MoonbaseLatencyTrackerRoot>
                    </MoonbaseRelayEnvironmentRoot>
                  </MoonbaseEventTrackerRoot>
                </CurrentUserRoot>
              </TachyonIntlRoot>
            </MoonbaseEventReporterRoot>
          </MoonbaseEnvironmentRoot>
        </DynamicSettingsRoot>
      </RouterUtilsRoot>
    </ClientSideErrorBoundary>
  );
};

MoonbaseApp.displayName = 'MoonbaseApp';

MoonbaseApp.getInitialProps = async (appContext) => {
  const { Component, ctx: tachyonCtx } = appContext;

  const persistedProps: MoonbasePersistedAppInitialProps = isBrowser()
    ? {
        /* eslint-disable @typescript-eslint/no-unsafe-assignment */
        appEnvironment: window.__NEXT_DATA__.props.appEnvironment,
        dynamicSettings: window.__NEXT_DATA__.props.dynamicSettings,
        experimentBucket: window.__NEXT_DATA__.props.experimentBucket,
        intlData: window.__NEXT_DATA__.props.intlData,
        language: window.__NEXT_DATA__.props.language,
        /* eslint-enable @typescript-eslint/no-unsafe-assignment */
      }
    : {
        /* eslint-disable @typescript-eslint/no-non-null-assertion */
        appEnvironment: tachyonCtx.req!.tachyon.appEnvironment,
        dynamicSettings: tachyonCtx.req!.tachyon.dynamicSettings,
        experimentBucket: tachyonCtx.req!.tachyon.experimentBucket,
        intlData: tachyonCtx.req!.tachyon.intlData,
        language: tachyonCtx.req!.tachyon.intlData.preferredLanguageTags[0],
        /* eslint-enable @typescript-eslint/no-non-null-assertion */
      };

  // update context before calling super.getInitialProps to make values
  // available to current page component
  tachyonCtx.language = persistedProps.language;
  const initialProps = await App.getInitialProps(appContext);

  const fetchQueryOpts: FetchQueryOpts = {
    language: persistedProps.language,
  };

  const relayQueryVariables: RelayQueryVariables =
    (initialProps.pageProps as TachyonPageInitialProps).queryVariables ?? {};

  const serverSideInitialProps =
    // typeof window and tachyonCtx.req/res should move in sync, here just for types
    typeof window === 'undefined' && isServersideContext(tachyonCtx)
      ? await getServersideInitialProps({
          Component,
          fetchQueryOpts: {},
          relayQueryVariables,
          tachyonCtx,
        })
      : undefined;

  return {
    ...initialProps,
    ...persistedProps,
    ...serverSideInitialProps,
    currentAsPath: tachyonCtx.asPath!, // eslint-disable-line @typescript-eslint/no-non-null-assertion
    fetchQueryOpts,
    relayQueryVariables,
  };
};
