import { lorem } from 'faker';
import { mockIntlData } from 'tachyon-intl';
import { RouterUtilsRoot } from 'tachyon-next-routing-utils';
import type {
  TachyonPage,
  TachyonPageInitialProps,
  TachyonPageProps,
} from 'tachyon-next-types';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { CoreText } from 'twitch-core-ui';
import { PageRenderer } from './PageRenderer';
import { getServersideInitialProps } from './getServersideInitialProps';
import type { MoonbaseAppProps } from '.';
import { MoonbaseApp } from '.';

jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  RequestInfoRoot: (props: any) => props.children,
}));

jest.mock('tachyon-event-tracker', () => ({
  ...jest.requireActual('tachyon-event-tracker'),
  Pageview: jest.fn(() => null),
}));

/* supress console messages when running tests */
jest.mock('tachyon-logger', () => ({ logger: { debug: jest.fn() } }));
const mockConsoleError = jest.spyOn(console, 'error').mockImplementation();
afterAll(() => {
  mockConsoleError.mockRestore();
});

const mockRouterUtilsRoot = RouterUtilsRoot as jest.Mock;
jest.mock('tachyon-next-routing-utils', () => {
  return {
    ...jest.requireActual('tachyon-next-routing-utils'),
    RouterUtilsRoot: jest.fn((props: any) => props.children),
  };
});

jest.mock('./configure', () => ({
  configure: jest.fn(),
}));

jest.mock('next/app', () => {
  const app = () => false;
  app.getInitialProps = () => ({ pageProps: { queryVariables: {} } });
  return {
    __esModule: true,
    default: app,
  };
});

jest.mock('tachyon-debug-reporter', () => ({
  configTachyonDebug: jest.fn(),
}));

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  EnvironmentRoot: (props: any) => props.children,
}));

jest.mock('./PageRenderer', () => ({
  ...jest.requireActual('./PageRenderer'),
  PageRenderer: jest.fn(({ Page }) => <Page />),
}));
const mockPageRenderer = PageRenderer as jest.Mock;

jest.mock('./getServersideInitialProps', () => ({
  ...jest.requireActual('./getServersideInitialProps'),
  getServersideInitialProps: jest.fn(() => ({ isLoggedInServerside: false })),
}));
const mockGetServersideInitialProps = getServersideInitialProps as jest.Mock;

const mockPage = (
  renderer: any = () => <div />,
): TachyonPage<TachyonPageInitialProps, TachyonPageProps> => {
  const page = renderer;
  page.pageviewTracking = defaultPageviewTracking;

  return page;
};

describe(MoonbaseApp, () => {
  const pageProps = { foo: 'bar' };
  const intlData = mockIntlData();

  const setup = createMountWrapperFactory(
    MoonbaseApp as any,
    (): MoonbaseAppProps => ({
      Component: mockPage(),
      appEnvironment: 'development',
      currentAsPath: '/',
      dynamicSettings: {
        environment: 'development',
        experiments: {},
        moonbaseDisabled: false,
        spadeUrl: '//spade',
      },
      experimentBucket: 'a',
      fetchQueryOpts: {},
      intlData,
      language: intlData.preferredLanguageTags[0],
      pageProps: pageProps as any,
      relayQueryVariables: {},
      router: {} as any,
    }),
  );

  it('renders <MinimalInternalError /> when there is an SSR error', () => {
    const { wrapper } = setup({ serversideError: true });

    expect(wrapper.find(CoreText).html()).toContain(
      'An unexpected error occurred, please refresh and try again.',
    );
  });

  it('renders the MoonbaseAppBoundary when the <Page/> fails', () => {
    const { wrapper } = setup({
      Component: mockPage(() => {
        throw new Error('oh no');
      }),
    });

    expect(wrapper.find({ boundaryName: 'MoonbaseAppBoundary' })).toExist();
  });

  it('renders the MoonbaseRootBoundary when the app fails', () => {
    mockRouterUtilsRoot.mockImplementationOnce(() => {
      throw new Error('oh no');
    });
    const { wrapper } = setup();

    expect(wrapper.find({ boundaryName: 'MoonbaseRootBoundary' })).toExist();
  });

  it('renders the <Page /> when there is not an SSR error', () => {
    const { props, wrapper } = setup();

    expect(wrapper.find(props.Component as any)).toExist();
  });

  it('renders the MoonbaseAppBoundary when the PageRenderer has an error', () => {
    mockPageRenderer.mockImplementationOnce(() => {
      throw new Error('oh no');
    });
    const { props, wrapper } = setup();

    expect(wrapper.find({ boundaryName: 'MoonbaseAppBoundary' })).toExist();
    expect(wrapper.find(props.Component as any)).not.toExist();
  });
});

describe('MoonbaseApp.getInitialProps', () => {
  describe('client side', () => {
    const mockNextData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
    };

    beforeEach(() => {
      global.window.__NEXT_DATA__ = { props: mockNextData } as any;
    });

    // @ts-expect-error: tests
    afterEach(() => delete global.window.__NEXT_DATA__);

    it('extracts __NEXT_DATA__ fields', async () => {
      const ip = await MoonbaseApp.getInitialProps({ ctx: {} } as any);

      expect(ip).toEqual(expect.objectContaining(mockNextData));
    });
  });

  describe('server side', () => {
    let mockWindow: any;
    beforeEach(() => {
      mockWindow = jest.spyOn(global as any, 'window', 'get');
      mockWindow.mockImplementation(() => undefined);
    });
    afterEach(() => {
      mockWindow.mockRestore();
    });

    const mockReqData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
      experimentBucket: lorem.word(),
      intlData: mockIntlData(),
    };

    const deviceId = lorem.word();
    const mockCtx = {
      ctx: { req: { tachyon: { ...mockReqData, deviceId } }, res: {} },
    };

    it('extracts req fields when', async () => {
      const ip = await MoonbaseApp.getInitialProps(mockCtx as any);

      expect(ip).toEqual(expect.objectContaining(mockReqData));
    });

    it('invokes getServersideInitialProps', async () => {
      await MoonbaseApp.getInitialProps(mockCtx as any);

      expect(mockGetServersideInitialProps).toHaveBeenCalledTimes(1);
    });
  });
});
