import type { DocumentInitialProps, DocumentProps } from 'next/document';
import Document, { Head, Html, Main, NextScript } from 'next/document';
import { ServerStyleSheet } from 'styled-components';
import { Fonts } from 'tachyon-more-ui';
import type { TachyonDocumentContext } from 'tachyon-next-types';
import type { MoonbaseRequestExtensions } from '../../../config';

export type MoonbaseDocumentInitialProps = DocumentInitialProps & {
  currentPathname: string;
  language: string;
};

export type MoonbaseDocumentProps = DocumentProps &
  MoonbaseDocumentInitialProps;

// this component is beginning to do webpack imports in order to enable plugins,
// which is really messing with Jest
// istanbul ignore next: not testable
/**
 * Root-level React component that is only rendered server-side, see:
 * https://nextjs.org/docs/#custom-document
 */
export class MoonbaseDocument extends Document<MoonbaseDocumentProps> {
  public static async getInitialProps(
    context: TachyonDocumentContext<MoonbaseRequestExtensions>,
  ): Promise<MoonbaseDocumentInitialProps> {
    const sheet = new ServerStyleSheet();
    const originalRenderPage = context.renderPage;

    try {
      context.renderPage = () =>
        originalRenderPage({
          enhanceApp: (App) => (props) =>
            sheet.collectStyles(<App {...props} />),
        });

      const initialProps = await super.getInitialProps(context);
      return {
        ...initialProps,
        currentPathname: context.pathname,
        language: context.req!.tachyon.intlData.preferredLanguageTags[0], // eslint-disable-line @typescript-eslint/no-non-null-assertion
        styles: sheet.getStyleElement(),
      };
    } finally {
      sheet.seal();
    }
  }

  public render(): JSX.Element {
    return (
      <Html lang={this.props.language}>
        <Head>
          <Fonts />
        </Head>
        <body>
          <Main />
          <NextScript />
        </body>
      </Html>
    );
  }
}
