/**
 * A list of all route names in the app. This list must manually be updated when a new
 * page is added to the application. The values map to the route's equivalent Spade "location".
 */
export enum RouteName {
  Error = 'error',
  External = 'external',
  HealthCheck = 'health_check',
  Homepage = 'homepage',
  NotFound = 'not_found',
  OAuth = 'oauth',
}

/**
 * A mapping of Route names to their Next equivalent static "pathname". These
 * map directly to the folder structure of src/pages/*.
 */
export const RouteLinkPathnames: { [key in RouteName]: string } = {
  [RouteName.Error]: '/_error',
  [RouteName.External]: '/external',
  [RouteName.HealthCheck]: '/_debug/running',
  [RouteName.Homepage]: '/',
  [RouteName.NotFound]: '/not-found',
  [RouteName.OAuth]: '/oauth',
  // catch-alls must be last in this list to allow for proper matching behavior
};

export type RouteLinkParams = {
  as?: string;
  href: string;
};

/**
 * A mapping of route names to a matching link builder function. The arguments
 * for these functions also serve as the definition for which "routeParams" to
 * provide to the app's "Link" component.
 *
 * DO NOT USE DIRECTLY: Use `Link` ("src/components/framework/Link").
 *
 * Rules:
 * 1) All functions must return an object of { href: string; as?: string }.
 * 1) Static routes should take no arguments in their function and return only an "href" field.
 * 2) Dynamic routes should return both an "href" (path with placeholder) and "as" (path with dynamic value) field.
 */
export const RouteLinkBuilders = {
  [RouteName.Error]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Error],
  }),
  // TODO: See if we can prune this out of here without upsetting our other
  // types. This will never be used in reality
  [RouteName.External]: ({ href }: { href: string }): RouteLinkParams => ({
    as: href,
    href,
  }),
  [RouteName.HealthCheck]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.HealthCheck],
  }),
  [RouteName.Homepage]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Homepage],
  }),
  [RouteName.NotFound]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.NotFound],
  }),
  [RouteName.OAuth]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.OAuth],
  }),
};

/**
 * Converts a custom RouteName to a Next pathname.
 * Returns the NotFound pathname if there is no matching route.
 */
export function pathnameFromRouteName(name: RouteName): string {
  return RouteLinkPathnames[name] || RouteLinkPathnames[RouteName.NotFound];
}

/**
 * Converts a Next pathname to a RouteName. Returns NotFound if there is no matching path.
 */
export function routeNameFromPathname(pathname: string): RouteName {
  const routeName = Object.keys(RouteLinkPathnames).find(
    (route) => RouteLinkPathnames[route as RouteName] === pathname,
  ) as RouteName | undefined;

  return routeName ?? RouteName.NotFound;
}
