import type { ParsedUrlQuery } from 'querystring';
import type { NextRouter } from 'next/router';
import { useRouter } from 'next/router';
import type { AsPathForRouteNameParams } from '../linkPartsForRouteName';
import { linkPartsForRouteName } from '../linkPartsForRouteName';
import type { RouteName } from '../routes';

export interface TachyonRouter {
  back(): void;
  push: <T extends RouteName>(
    params: AsPathForRouteNameParams<T>,
    query?: ParsedUrlQuery,
  ) => ReturnType<NextRouter['push']>;
  reload(): void;
  replace: <T extends RouteName>(
    params: AsPathForRouteNameParams<T>,
    query?: ParsedUrlQuery,
  ) => ReturnType<NextRouter['replace']>;
}

export function useTachyonRouter(): TachyonRouter {
  const router = useRouter();

  function push<T extends RouteName>(
    params: AsPathForRouteNameParams<T>,
    query?: ParsedUrlQuery,
  ): ReturnType<NextRouter['push']> {
    const { as, href } = linkPartsForRouteName(params, query);
    return router.push(href, as);
  }

  function replace<T extends RouteName>(
    params: AsPathForRouteNameParams<T>,
    query?: ParsedUrlQuery,
  ): ReturnType<NextRouter['replace']> {
    const { as, href } = linkPartsForRouteName(params, query);
    return router.replace(href, as);
  }

  function back() {
    router.back();
  }

  function reload() {
    // shallow: false causes the page to run through the full getInitialProps flow again
    router.push(router.pathname, router.asPath, { shallow: false });
  }

  return {
    back,
    push,
    reload,
    replace,
  };
}
