import { datatype, internet, lorem, random } from 'faker';
import type { IntlData } from 'tachyon-intl';
import { selectIntlData } from 'tachyon-intl-server';
import type {
  TachyonRequest,
  TachyonRequestExtension,
  TachyonResponse,
} from 'tachyon-next-types';
import { createDeviceIDOnServer } from 'tachyon-server-utils';
import { GLOBAL_COOKIE_NAMES, TACHYON_DEVICE_ID_PREFIX } from 'tachyon-utils';
import {
  EXPERIMENT_BUCKET_HEADER,
  EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY,
  USER_LANGUAGE_COOKIE_NAME,
} from '../../config';
import { DEBUG_PATH_PREFIX, createAppRequestHandler } from '.';

const mockSpadeUrl = 'https://spade.twitch.tv/';
const mockLocale = random.locale();
jest.mock('accepts', () => () => ({
  languages: () => [mockLocale],
}));

const mockIntlData: IntlData = {
  locales: [],
  preferredLanguageTags: [mockLocale],
};
jest.mock('tachyon-intl-server', () => ({
  selectIntlData: jest.fn(() => mockIntlData),
}));
const mockSelectIntlData = selectIntlData as jest.Mock;

jest.mock('tachyon-server-utils', () => ({
  ...jest.requireActual('tachyon-server-utils'),
  createDeviceIDOnServer: jest.fn(() => 'tach-fake-device-id'),
}));
const mockCreateDeviceIDOnServer = createDeviceIDOnServer as jest.Mock;

describe(createAppRequestHandler, () => {
  const mockContext = {
    appEnvironment: 'development',
    dynamicSettings: {
      spadeUrl: mockSpadeUrl,
    },
  } as TachyonRequestExtension;
  const deviceId = datatype.uuid();
  const path = `/${lorem
    .words(datatype.number({ max: 4 }))
    .replace(/ /g, '/')}`;
  const experimentBucket = 'f';
  const experimentOverrides = datatype.uuid();

  let req: TachyonRequest;
  let res: TachyonResponse;
  const next = () => undefined;
  const mockNextRequestHandler = jest.fn();
  const handler = createAppRequestHandler(mockContext, mockNextRequestHandler);

  beforeEach(() => {
    mockNextRequestHandler.mockReset();

    req = {
      cookies: {
        [GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME]: deviceId,
      },
      headers: {
        [EXPERIMENT_BUCKET_HEADER]: experimentBucket,
      },
      path,
      query: {
        [EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY]: experimentOverrides,
      },
      url: internet.url(),
    } as any;

    res = { cookie: jest.fn() } as any;
  });

  it('creates a request handler function that adds static properties to req and calls next request handler', () => {
    expect(handler).toBeInstanceOf(Function);

    handler(req, res, next);
    expect(req.tachyon).toEqual(
      expect.objectContaining({
        appEnvironment: 'development',
        dynamicSettings: { spadeUrl: mockSpadeUrl },
        intlData: mockIntlData,
      }),
    );

    expect(mockSelectIntlData).toHaveBeenLastCalledWith([mockLocale]);
    expect(mockNextRequestHandler).toHaveBeenCalledWith(
      req,
      res,
      expect.objectContaining({ href: expect.stringContaining(req.url) }),
    );
  });

  describe('languages', () => {
    it('finds language in headers', () => {
      req.query['accept-language'] = mockLocale;

      handler(req, res, next);
      expect(mockSelectIntlData).toHaveBeenLastCalledWith([mockLocale]);
    });

    it('finds language in cookies, which is language from users-service', () => {
      const language = 'fr';
      req.cookies[USER_LANGUAGE_COOKIE_NAME] = language;

      handler(req, res, next);
      expect(mockSelectIntlData).toHaveBeenLastCalledWith([
        language,
        mockLocale,
      ]);
    });

    it('overrides language in request with that from users-service stored in cookies', () => {
      const userLang = 'fr';
      req.query['accept-language'] = mockLocale;
      req.cookies[USER_LANGUAGE_COOKIE_NAME] = userLang;

      handler(req, res, next);
      expect(mockSelectIntlData).toHaveBeenLastCalledWith([
        userLang,
        mockLocale,
      ]);
    });
  });

  describe('device id', () => {
    it('finds an existing device id and does not warn', () => {
      handler(req, res, next);

      expect(req.tachyon).toEqual(
        expect.objectContaining({
          deviceId,
        }),
      );

      expect(mockCreateDeviceIDOnServer).not.toHaveBeenCalled();
    });

    it('creates a missing device id when missing from the request and warns', () => {
      delete req.cookies[GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME];
      handler(req, res, next);

      expect(req.tachyon).toEqual(
        expect.objectContaining({
          deviceId: expect.stringContaining(TACHYON_DEVICE_ID_PREFIX),
        }),
      );
      expect(mockCreateDeviceIDOnServer).toHaveBeenCalled();
    });

    it('does nothing for debug paths', () => {
      delete req.cookies[GLOBAL_COOKIE_NAMES.DEVICE_ID_COOKIE_NAME];
      req.path = DEBUG_PATH_PREFIX;

      handler(req, res, next);

      expect(req.tachyon).toEqual(
        expect.not.objectContaining({
          deviceId: expect.anything(),
        }),
      );
      expect(mockCreateDeviceIDOnServer).not.toHaveBeenCalled();
    });
  });
});
