import { resolve } from 'path';
import { URL } from 'url';
import cookieParser from 'cookie-parser';
import express from 'express';
import next from 'next';
import open from 'open';
import { fetchDynamicSettings } from 'tachyon-dynamic-settings';
import { prepareIntlDataCache } from 'tachyon-intl-server';
import {
  getTachyonEnvVar,
  tachyonServerMiddleware,
} from 'tachyon-server-utils';
import { ensureBoolean } from 'tachyon-utils';
import type { MoonbaseDynamicSettings } from '../config';
import { RouteName, dynamicPathnames, pathnameFromRouteName } from '../routing';
import { createAppRequestHandler } from './appHandler';
import { isDevelopment } from './utils';

// polyfill WHATWG URL
// eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
(global as any).URL = URL;

const PORT = process.env.PORT ? parseInt(process.env.PORT, 10) : 3400;
const dev = isDevelopment();

// from `dist/server/index.js` to dir containing `.next/` & `messages/`
const appDir = dev ? undefined : resolve(__dirname, '../..');

// 15 minutes
const DYNAMIC_SETTINGS_TTL = 15 * 60 * 1000;

const app = next({ dev, dir: appDir });
const nextRequestHandler = app.getRequestHandler();
const appEnvironment = getTachyonEnvVar();

// istanbul ignore next: trivial + type-based enforcement
const fetchMoonbaseDynamicSettings =
  fetchDynamicSettings<MoonbaseDynamicSettings>({
    app: 'moonbase',
    appEnvironment,
    appGroup: 'tachyon',
    processor: ({ moonbase_disabled, ...settings }) => {
      // fallback to disabled for safety
      const moonbaseDisabled = ensureBoolean(moonbase_disabled, true);

      return {
        ...settings,
        moonbaseDisabled,
      };
    },
  });

// top-level await unavailable until Node 14.8
// istanbul ignore next: trivial
prepareIntlDataCache({
  appDir,
})
  .then(async () => {
    const dynamicSettings = await fetchMoonbaseDynamicSettings();
    const ctx = {
      appEnvironment,
      dynamicSettings,
    };

    setInterval(() => {
      fetchMoonbaseDynamicSettings().then((settings) => {
        ctx.dynamicSettings = settings;
      });
    }, DYNAMIC_SETTINGS_TTL).unref();

    return ctx;
  })
  .then((ctx) =>
    app.prepare().then(() => {
      const server = express();
      // Necessary until https://github.com/vercel/next.js/issues/17096 is fixed
      server.use(
        tachyonServerMiddleware({
          badRequestMiddleware: {
            badRequestRedirectPath: pathnameFromRouteName(RouteName.NotFound),
            dynamicPathnames: dynamicPathnames(),
          },
        }),
      );
      server.use(cookieParser());
      // tell express to populate `req` convenience attributes from X-Forwarded-* headers
      // http://expressjs.com/en/4x/api.html#req.hostname
      server.enable('trust proxy');

      const appRequestHandler = createAppRequestHandler(
        ctx,
        nextRequestHandler,
      );
      // api routes are the only routes that need to handle POST requests, so we
      // make this explicit to minimize our exposed surface area
      server.post('/api/*', appRequestHandler);
      server.get('*', appRequestHandler);

      server.listen(PORT, () => {
        if (dev) {
          const devUrl = 'https://localhost.appeals.twitch.tv';
          console.log(`\nListening at ${devUrl}\n`);
          open(devUrl);
        }
      });
    }),
  )
  .catch((error: Error) => {
    console.error(error.stack);
    process.exit(1);
  });
