import { readdirSync, statSync } from 'fs';
import { extname, join, relative, resolve } from 'path';
import { staticRoutePrefix } from '../env.config';

/**
 * The path prefix for proper compatibility in GH pages
 */
const PATH_PREFIX = `${staticRoutePrefix}/d/`;

/**
 * File extensions that we expect to be Markdown and will generate doc pages for.
 */
const MARKDOWN_DOC_EXTENSIONS = ['.md'];

/**
 * A list of directories we don't want to include in our doc generator.
 */
const PATH_IGNORE_LIST = ['node_modules', 'pull_request_template.md', '.git'];

/**
 * Root of the Tachyon repo.
 */
const TACHYON_ROOT = join(__dirname, '../../../');

/**
 * Checks if a given path is a directory. If so, makes a recursive call. Otherwise
 * returns a path if the file is a Markdown extension.
 */
function parseSubDirectory(currentDir: string, subDir: string): string[] {
  if (PATH_IGNORE_LIST.includes(subDir)) {
    return [];
  }

  const subDirectoryPath = resolve(currentDir, subDir);

  const dirStat = statSync(subDirectoryPath);
  if (dirStat.isDirectory()) {
    return getMarkdownFilePaths(subDirectoryPath);
  }

  if (MARKDOWN_DOC_EXTENSIONS.indexOf(extname(subDirectoryPath)) > -1) {
    return [subDirectoryPath];
  }

  return [];
}

function makePathsRelative(absPathFiles: string[]): string[] {
  return absPathFiles.map((newFile) => relative(TACHYON_ROOT, newFile));
}

function getMarkdownFilePaths(rootDirectoryPath: string): string[] {
  const tachyonProjectDirs = readdirSync(rootDirectoryPath);
  const dirParsers = tachyonProjectDirs.map(dirMDFinder(rootDirectoryPath));

  // Squash all the results into a single array
  return dirParsers.reduce((mdFilePaths, newFilePaths) => {
    return mdFilePaths.concat(newFilePaths);
  }, []);
}

function dirMDFinder(currentDir: string): (subDir: string) => string[] {
  return (subDir: string) => parseSubDirectory(currentDir, subDir);
}

function findDocPaths(): string[] {
  const absPathFiles = getMarkdownFilePaths(TACHYON_ROOT);

  const relativePathFiles = makePathsRelative(absPathFiles);
  console.log('Found files:');
  relativePathFiles.forEach((file) => console.log(`- ${file}`));
  return relativePathFiles;
}

interface NextExportMeta {
  page: string;
  query: {
    docFilePath: string;
  };
}

type NextExportMapping = Record<string, NextExportMeta>;

export function buildDocToPathMapping(): NextExportMapping {
  return findDocPaths().reduce<NextExportMapping>((acc, docFilePath) => {
    const pathData = {
      page: '/',
      query: { docFilePath: `/${docFilePath}` },
    };

    if (docFilePath === 'README.md') {
      /*
        root README is special in that it must also correspond to the unprefixed
        root index.html and the index.html of the PATH_PREFIX
      */
      acc['/'] = pathData;
      acc[PATH_PREFIX] = pathData;
    } else {
      /*
        READMEs are treated as the index.html of their enclosing directory
        and all other file paths are mapped directly. Removes trailing
        `/README.md`/`.md` because a) otherwise Next will try to return a
        non-HTML content-type and b) because we normalize all app paths into
        `/`-terminated strings for URL consistency and Github compatibility.
      */
      const appPath = `${PATH_PREFIX}${docFilePath.replace(
        /(?:\/README)?\.md$/,
        '',
      )}/`;
      acc[appPath] = pathData;
    }

    return acc;
  }, {});
}

function run(): void {
  console.log('Searching Tachyon using project root: ', TACHYON_ROOT);
  const docFilePathMapping = buildDocToPathMapping();
  console.log('\nPath maps\n', docFilePathMapping);
}

// Runs if not being imported
if (!module.parent) {
  run();
}
