import NextLink from 'next/link';
import type { FC } from 'react';
import { useMemo, useRef, useState } from 'react';
import {
  BalloonSize,
  BorderRadius,
  Color,
  CoreText,
  DropDownMenu,
  FontSize,
  FontWeight,
  Interactable,
  InteractableType,
  Layout,
  SearchInput,
} from 'twitch-core-ui';
import {
  filterCoreUiLinkProps,
  getDocPageHref,
  getDocPageLink,
  search,
} from '../../../utils';

export const Search: FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const results = useMemo(
    () => (searchTerm === '' ? [] : search(searchTerm)),
    [searchTerm],
  );
  const dropDownAnchorRef = useRef<HTMLInputElement | null>(null);

  return (
    <>
      <SearchInput
        onChange={(e) => {
          setSearchTerm(e.currentTarget.value);
        }}
        ref={dropDownAnchorRef}
        value={searchTerm}
      />
      <DropDownMenu
        anchorRef={dropDownAnchorRef.current}
        disableAriaHideApp
        disableFocusAfterRender
        onRequestClose={() => undefined}
        options={{
          placement: 'bottom-start',
        }}
        show={results.length > 0}
        size={BalloonSize.Auto}
      >
        {results.map((r) => (
          <Interactable
            borderRadius={BorderRadius.Medium}
            key={r.urlPath}
            linkTo="/deferToRenderLink"
            onClick={() => {
              setSearchTerm('');
            }}
            renderLink={(props) => (
              <NextLink
                as={getDocPageLink(r.urlPath)}
                href={getDocPageHref(r.filePath)}
              >
                <a {...filterCoreUiLinkProps(props)} />
              </NextLink>
            )}
            variant={InteractableType.Alpha}
          >
            <Layout padding={{ x: 1, y: 0.5 }}>
              <CoreText
                fontSize={FontSize.Size5}
                fontWeight={FontWeight.SemiBold}
              >
                {r.title}
              </CoreText>
              <CoreText color={Color.Alt} fontSize={FontSize.Size6}>
                {r.urlPath}
              </CoreText>
            </Layout>
          </Interactable>
        ))}
      </DropDownMenu>
    </>
  );
};

Search.displayName = 'Search';
