import NextLink from 'next/link';
import { Component } from 'react';
import {
  AlignItems,
  Background,
  BorderRadius,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  InjectLayout,
  Interactable,
  JustifyContent,
  Layout,
  SVG,
  SVGAsset,
  TextDecoration,
  Tooltip,
} from 'twitch-core-ui';
import type { Topic } from '../../../../utils';
import {
  filterCoreUiLinkProps,
  getDocPageHref,
  getDocPageLink,
} from '../../../../utils';

export interface NavSectionProps {
  forceExpanded?: boolean;
  pageDocFilePath: string;
  root?: boolean;
  topic: Topic;
}

interface NavSectionState {
  expanded: boolean;
}

export class NavSection extends Component<NavSectionProps, NavSectionState> {
  public state: NavSectionState = {
    expanded:
      this.props.pageDocFilePath === this.props.topic.document?.filePath,
  };

  public static displayName = 'NavSection';

  public render(): JSX.Element {
    return (
      <>
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexDirection={FlexDirection.Row}
          flexShrink={0}
          justifyContent={JustifyContent.Between}
        >
          {this.renderNavHeader()}
          {this.renderShowMoreIcon()}
        </Layout>
        {this.renderNavSections()}
      </>
    );
  }

  private renderShowMoreIcon(): JSX.Element | null {
    const { subTopics, title } = this.props.topic;

    if (this.props.forceExpanded || Object.keys(subTopics).length === 0) {
      return null;
    }

    const label = this.state.expanded
      ? `Hide ${title} topics`
      : `Show more ${title} topics`;

    return (
      <Layout margin={{ right: 0.5 }}>
        <Interactable onClick={this.handleShowMoreIconClick}>
          <Tooltip label={label}>
            <SVG
              asset={
                this.state.expanded ? SVGAsset.AngleUp : SVGAsset.AngleDown
              }
            />
          </Tooltip>
        </Interactable>
      </Layout>
    );
  }

  private renderNavHeader(): JSX.Element {
    const {
      pageDocFilePath,
      root,
      topic: { document, title },
    } = this.props;
    const active = pageDocFilePath === document?.filePath;

    const styledTitle = (
      <InjectLayout
        background={active ? Background.Alt2 : undefined}
        padding={{ x: 1, y: 0.5 }}
      >
        <CoreText
          bold={root}
          decoration={active ? TextDecoration.Underline : undefined}
          fontSize={FontSize.Size5}
        >
          {title}
        </CoreText>
      </InjectLayout>
    );

    if (document) {
      // All links go to / to satisfy client-side next, with the docPath query
      // value provided in the same manner that exportPathMap does for
      // server-side next. We use the `as` prop to generate "vanity" URLs for
      // the browser url state and HTML that are properly handled by the
      // server/github.
      return (
        <Interactable
          borderRadius={BorderRadius.Medium}
          linkTo="/deferToRenderLink"
          onClick={this.handleHeaderClick}
          renderLink={(props) => (
            <NextLink
              as={getDocPageLink(document.urlPath)}
              href={getDocPageHref(document.filePath)}
            >
              <a {...filterCoreUiLinkProps(props)} />
            </NextLink>
          )}
        >
          {styledTitle}
        </Interactable>
      );
    }

    return styledTitle;
  }

  private renderNavSections = (): JSX.Element[] | null => {
    if (!this.props.forceExpanded && !this.state.expanded) {
      return null;
    }

    return Object.entries(this.props.topic.subTopics).map(
      ([subTopicName, subTopicData]) => (
        <Layout fullWidth key={subTopicName} padding={{ left: 1 }}>
          <NavSection
            pageDocFilePath={this.props.pageDocFilePath}
            topic={subTopicData}
          />
        </Layout>
      ),
    );
  };

  private handleHeaderClick = (): void => {
    this.setState({ expanded: true });
  };

  private handleShowMoreIconClick = (): void => {
    this.setState(({ expanded }) => ({
      expanded: !expanded,
    }));
  };
}
