import type { FC } from 'react';
import { memo } from 'react';
import type { ReactMarkdownOptions } from 'react-markdown';
import ReactMarkdown, { uriTransformer } from 'react-markdown';
import rehypeRaw from 'rehype-raw';
import gfm from 'remark-gfm';
import {
  CoreImage,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  Typeset,
} from 'twitch-core-ui';
import { getDocPageLink, getDocuments, getStaticSrc } from '../../../utils';
import { CodeBlock } from '../CodeBlock';
import { Header } from '../Header';

export interface StyledMarkdownProps {
  markdown: string;
}

const DOC_PORTAL_IMAGE_TEST = /^\.(?:[\w-]|\/|\.)*static\//;

type ImageURITransformer = NonNullable<
  ReactMarkdownOptions['transformImageUri']
>;

/**
 * Maps image URIs to the doc portal's `/static/` folder if the URI is relative
 * since t
 * @param uri
 */
export const imageURITransformer: ImageURITransformer = (uri, _alt, _title) => {
  if (DOC_PORTAL_IMAGE_TEST.exec(uri)) {
    return uri.replace(DOC_PORTAL_IMAGE_TEST, getStaticSrc(''));
  }

  return uri;
};

type LinkURITransformer = Exclude<
  NonNullable<ReactMarkdownOptions['transformLinkUri']>,
  boolean
>;

// Handles relative links and links to page sections (e.g. ./readme, ../readme, #section).
const DOC_LOCAL_LINK_TARGET_TEST = /^(\.+)\/|^#/;

type LinkTargetResolver = Exclude<
  NonNullable<ReactMarkdownOptions['linkTarget']>,
  string
>;

/**
 * Function for determining what "target" value to set for a Markdown link.
 * Use same tab when linking to other documentation pages, otherwise open a
 * new tab.
 * @param uri
 */
export const linkTargetResolver: LinkTargetResolver = (
  uri,
  _children,
  _title,
) => {
  if (DOC_LOCAL_LINK_TARGET_TEST.exec(uri)) {
    return '_self';
  }

  return '_blank';
};

/**
 * Converts relative links in markdown files to their corresponding location in the docs.
 * @param uri
 */
export const linkURITransformer: LinkURITransformer = (
  uri,
  _children,
  _title,
) => {
  const transformedURI = uriTransformer(uri);

  if (transformedURI.startsWith('.')) {
    // if a link starts with a . it is a relative link and we compare it with
    // all of the known doc paths to find a match and then replace it with
    // an absolute path instead
    const prefixFreeURIPath = transformedURI.replace(/^[./]*/, '');
    const document = getDocuments().find((doc) =>
      doc.filePath.endsWith(prefixFreeURIPath),
    );

    if (document) {
      return getDocPageLink(document.urlPath);
    }
  }

  return transformedURI;
};

const ConstrainedImage: FC = (props) => (
  <CoreImage style={{ maxWidth: '50%' }} {...(props as any)} />
);
ConstrainedImage.displayName = 'ConstrainedImage';

// we don't need to render <pre> tags because syntax highlighter takes care of
// that for us
const NoopPre: FC = ({ children }) => <>{children}</>;
NoopPre.displayName = 'NoopPre';

// See https://github.com/rexxars/react-markdown/blob/master/src/renderers.js for example renderers
export const StyledMarkdownBase: FC<StyledMarkdownProps> = ({ markdown }) => (
  <Typeset>
    <ReactMarkdown
      children={markdown}
      components={{
        // types aren't perfect here but so far they work, and no point in
        // bothering Core UI about them yet
        /* eslint-disable @typescript-eslint/no-unsafe-assignment */
        code: CodeBlock as any,
        h1: Header as any,
        h2: Header as any,
        h3: Header as any,
        h4: Header as any,
        h5: Header as any,
        h6: Header as any,
        /* eslint-enable @typescript-eslint/no-unsafe-assignment */
        img: ConstrainedImage,
        pre: NoopPre,
        table: Table,
        tbody: TableBody,
        /* eslint-disable @typescript-eslint/no-unsafe-assignment */
        td: TableCell as any,
        th: TableHeading as any,
        /* eslint-enable @typescript-eslint/no-unsafe-assignment */
        thead: TableHeader,
        tr: TableRow,
      }}
      linkTarget={linkTargetResolver}
      plugins={[gfm]}
      rehypePlugins={[rehypeRaw]}
      transformImageUri={imageURITransformer}
      transformLinkUri={linkURITransformer}
    />
  </Typeset>
);
/* eslint-enable react/display-name,react/forbid-dom-props */

StyledMarkdownBase.displayName = 'StyledMarkdownBase';

export const StyledMarkdown = memo(StyledMarkdownBase);

StyledMarkdown.displayName = 'StyledMarkdown';
