import { getDocPageLink, getDocuments, getStaticSrc } from '../../../utils';
import {
  StyledMarkdownBase,
  imageURITransformer,
  linkTargetResolver,
  linkURITransformer,
} from '.';

jest.mock('../../../utils', () => ({
  ...jest.requireActual('../../../utils'),
  getDocuments: jest.fn(),
}));
const getDocAppPathsMock = getDocuments as jest.Mock;

describe(StyledMarkdownBase, () => {
  describe(imageURITransformer, () => {
    it('trims relative tachyon doc paths down to work with the static folder', () => {
      expect(
        imageURITransformer('../../apps/sky-map/static/foo/bar.png', '', null),
      ).toEqual(getStaticSrc('/foo/bar.png'));

      expect(imageURITransformer('../static/foo/bar.png', '', null)).toEqual(
        getStaticSrc('/foo/bar.png'),
      );
    });

    it('does not rewrite non-relative image paths', () => {
      expect(
        imageURITransformer('https://some/image/bar.png', '', null),
      ).toEqual('https://some/image/bar.png');
    });

    it('opens links within the documentation portal in the same tab', () => {
      expect(linkTargetResolver('./apps/tomorrow/README', [], null)).toEqual(
        '_self',
      );
      expect(
        linkTargetResolver('#informative-documentation-section', [], null),
      ).toEqual('_self');
    });

    it('opens external links in a new tab', () => {
      expect(linkTargetResolver('https://some/link/bar', [], null)).toEqual(
        '_blank',
      );
    });

    describe('converts relative links in markdown files to correct documentation path', () => {
      beforeEach(() => {
        getDocAppPathsMock.mockImplementationOnce(() => [
          // data values copied from runtime
          {
            filePath: '/apps/tomorrow/processes/device-testing.md',
            urlPath: '/apps/tomorrow/processes/device-testing/',
          },
          {
            filePath: '/packages/tachyon-core/latency-tracker/README.md',
            urlPath: '/packages/tachyon-core/latency-tracker/',
          },
          {
            filePath: '/packages/README.md',
            urlPath: '/packages/',
          },
        ]);
      });

      it('when target is non-README markdown', () => {
        expect(
          linkURITransformer('./processes/device-testing.md', [], null),
        ).toEqual(getDocPageLink('/apps/tomorrow/processes/device-testing/'));
      });

      it('when target corresponds to nested README markdown', () => {
        expect(
          linkURITransformer('./latency-tracker/README.md', [], null),
        ).toEqual(getDocPageLink('/packages/tachyon-core/latency-tracker/'));
      });

      it('when target corresponds to root README markdown', () => {
        expect(linkURITransformer('./packages/README.md', [], null)).toEqual(
          getDocPageLink('/packages/'),
        );
      });
    });
  });
});
