import type { NextPage } from 'next';
import Head from 'next/head';
import { AlignItems, Display, JustifyContent, Layout } from 'twitch-core-ui';
import { requireDocument } from '../../../utils';
import {
  GithubLink,
  GithubLinkType,
  Search,
  StyledMarkdown,
} from '../../common';

export interface DocumentationPageProps {
  docFilePath: string;
  pageMarkdown: string;
}

export const DEFAULT_TITLE = 'Tachyon Docs';

export const DocumentationPage: NextPage<DocumentationPageProps> = ({
  docFilePath,
  pageMarkdown,
}) => {
  if (!pageMarkdown) {
    return (
      <>
        <Head>
          <title>{`${DEFAULT_TITLE} - Not Found`}</title>
        </Head>
        <div>No documentation found!</div>
      </>
    );
  }

  const pageTitle = /^# ([\w- ]+)/.exec(pageMarkdown);
  const title = pageTitle?.[1]
    ? `${DEFAULT_TITLE} - ${pageTitle[1]}`
    : DEFAULT_TITLE;

  return (
    <>
      <Head>
        <title>{title}</title>
      </Head>
      <Layout as="main" fullHeight>
        <Layout padding={{ bottom: 5, top: 1, x: 3 }}>
          <Layout
            alignItems={AlignItems.Center}
            display={Display.Flex}
            justifyContent={JustifyContent.End}
            padding={{ bottom: 1 }}
          >
            <Layout padding={{ right: 1 }}>
              <Search />
            </Layout>
            <Layout padding={{ x: 0.5 }}>
              <GithubLink
                docFilePath={docFilePath}
                type={GithubLinkType.Edit}
              />
            </Layout>
            <Layout padding={{ x: 0.5 }}>
              <GithubLink
                docFilePath={docFilePath}
                type={GithubLinkType.View}
              />
            </Layout>
          </Layout>
          <StyledMarkdown markdown={pageMarkdown} />
        </Layout>
      </Layout>
    </>
  );
};

DocumentationPage.displayName = 'DocumentationPage';

DocumentationPage.getInitialProps = (ctx) => {
  // docFilePath is baked into each page's query object from the exportPathMap
  // process in next.config.js at build time during server/export and explicitly
  // provided by all in-app links client-side
  const docFilePath = ctx.query.docFilePath as string;
  const documentMarkdown = requireDocument(docFilePath);

  return {
    docFilePath,
    pageMarkdown: documentMarkdown,
  };
};
