import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { requireDocument } from '../../../utils';
import { StyledMarkdown } from '../../common';
import { DEFAULT_TITLE, DocumentationPage } from '.';

jest.mock('../../../utils', () => ({
  requireDocument: jest.fn(),
}));

const requireDocumentMock = requireDocument as jest.Mock;

describe(DocumentationPage, () => {
  const docFilePath = '/README.md';
  const pageMarkdown = `# Markdown Title

    This is a Markdown file.`;

  describe('static methods', () => {
    beforeEach(() => {
      requireDocumentMock.mockReset();
    });

    it('getInitialProps requires md for a doc path and returns correct props', async () => {
      requireDocumentMock.mockReturnValueOnce(pageMarkdown);
      const result = await DocumentationPage.getInitialProps!({
        query: { docFilePath },
      } as any);

      expect(requireDocumentMock).toHaveBeenCalledWith(docFilePath);
      expect(result).toEqual({
        docFilePath,
        pageMarkdown,
      });
    });
  });

  describe('render', () => {
    const setup = createShallowWrapperFactory(DocumentationPage, () => ({
      docFilePath,
      pageMarkdown,
    }));

    it('renders extended title and text', () => {
      const { wrapper } = setup();
      expect(wrapper.find('title').text()).toEqual(
        `${DEFAULT_TITLE} - Markdown Title`,
      );
      expect(wrapper.find(StyledMarkdown)).toHaveProp({
        markdown: pageMarkdown,
      });
    });

    it('renders default title for poorly formatted markdown', () => {
      const { props, wrapper } = setup({ pageMarkdown: `#${pageMarkdown}` });
      expect(wrapper.find('title').text()).toEqual(`${DEFAULT_TITLE}`);
      expect(wrapper.find(StyledMarkdown)).toHaveProp({
        markdown: props.pageMarkdown,
      });
    });

    it('renders not found when not given markdown', () => {
      const { wrapper } = setup({ pageMarkdown: undefined });
      expect(wrapper.find('title').text()).toEqual(
        `${DEFAULT_TITLE} - Not Found`,
      );
      expect(wrapper.find('div').text()).toContain('No documentation found');
    });
  });
});
