import { getLinkPrefix } from '../getEnvVar';

export const DOC_PAGE_ROUTE_PREFIX = 'd/';
export const ENV_PROCESS_LINK_PREFIX_KEY = 'NEXT_ASSET_PREFIX';

/**
 * Util function to simplify joining two paths together.
 *
 * @param firstPart
 * @param secondPart
 */
export function joinPath(firstPart: string, secondPart: string): string {
  const formattedFirstPart = firstPart.replace(/\/$/, '');
  const formattedSecondPart = secondPart.replace(/^\//, '');

  return `${formattedFirstPart}/${formattedSecondPart}`;
}

export function getIndexLink(): string {
  const linkPrefixFromEnv = getLinkPrefix();
  if (
    process.env.NODE_ENV !== 'test' &&
    typeof linkPrefixFromEnv === 'undefined'
  ) {
    throw new Error(
      `No process.env value found for key: ${ENV_PROCESS_LINK_PREFIX_KEY}`,
    );
  }

  // We set a default to make testing less hairy. Otherwise we'd have to
  // mock process.env values for all upstream consumers which is painful
  const linkPrefix = linkPrefixFromEnv ?? '';

  return joinPath('/', linkPrefix.replace(/\/$/, ''));
}

export function getStaticSrc(relativeSrc: string): string {
  if (process.env.NODE_ENV === 'development') {
    return joinPath('/static', relativeSrc);
  }

  return joinPath(getIndexLink(), joinPath('static', relativeSrc));
}

export function getDocPageLinkRoot(): string {
  return joinPath(getIndexLink(), DOC_PAGE_ROUTE_PREFIX);
}

export function getDocPageLink(appPath: string): string {
  const linkRoot = getDocPageLinkRoot();

  if (appPath === '/') {
    return getIndexLink();
  }

  return joinPath(linkRoot, appPath);
}

/**
 * generates link values for use by next's router for in-app navigation
 */
export function getDocPageHref(docFilePath: string): string {
  return `/?docFilePath=${encodeURIComponent(docFilePath)}`;
}

export function getDocPathRootHref(): string {
  return getDocPageHref('/README.md');
}
