import { getLinkPrefix } from '../getEnvVar';
import {
  DOC_PAGE_ROUTE_PREFIX,
  getDocPageHref,
  getDocPageLink,
  getDocPageLinkRoot,
  getDocPathRootHref,
  getIndexLink,
  getStaticSrc,
  joinPath,
} from '.';

jest.mock('../getEnvVar', () => ({
  getLinkPrefix: jest.fn(),
}));
const getLinkPrefixMock = getLinkPrefix as jest.Mock;

describe('Route tests', () => {
  beforeEach(() => {
    getLinkPrefixMock.mockReturnValue('');
  });

  afterEach(() => {
    getLinkPrefixMock.mockReset();
  });

  describe(joinPath, () => {
    it('joins two paths as expected', () => {
      expect(joinPath('foo', 'bar')).toEqual('foo/bar');
      expect(joinPath('foo/', '/bar')).toEqual('foo/bar');
      expect(joinPath('foo/', 'bar')).toEqual('foo/bar');
      expect(joinPath('foo', '/bar')).toEqual('foo/bar');
    });
  });

  describe(getIndexLink, () => {
    it('returns "/" if no env variable is defined', () => {
      expect(getIndexLink()).toEqual('/');
    });

    it('returns /<prefix> if an env variable is defined', () => {
      getLinkPrefixMock.mockReturnValue('some/next/asset/prefix/');
      expect(getIndexLink()).toEqual('/some/next/asset/prefix');
    });
  });

  describe(getDocPageLinkRoot, () => {
    it('prefixes an env variable if present', () => {
      expect(getDocPageLinkRoot()).toEqual(
        `${getIndexLink()}${DOC_PAGE_ROUTE_PREFIX}`,
      );

      getLinkPrefixMock.mockReturnValue('envValue');
      expect(getDocPageLinkRoot()).toEqual(
        `/envValue/${DOC_PAGE_ROUTE_PREFIX}`,
      );
    });
  });

  describe(getDocPageLink, () => {
    it('assembles a valid URL path that is indexLink prefixed', () => {
      expect(getDocPageLink('/foo/bar')).toEqual(`${getIndexLink()}d/foo/bar`);
    });

    it('assembles a valid URL path that is not / prefixed', () => {
      expect(getDocPageLink('foo/bar')).toEqual(`${getIndexLink()}d/foo/bar`);
    });

    it('adds NEXT_ASSET_PREFIX to links when it is defined', () => {
      getLinkPrefixMock.mockReturnValue('some/next/asset/prefix');

      expect(getDocPageLink('to/some/place')).toEqual(
        '/some/next/asset/prefix/d/to/some/place',
      );
    });
  });

  describe(getStaticSrc, () => {
    it('assembles a valid static src when no assetPrefix is defined', () => {
      expect(getStaticSrc('/foo')).toEqual('/static/foo');
      expect(getStaticSrc('')).toEqual('/static/');
    });

    it('assembles a valid static src when an assetPrefix is defined', () => {
      getLinkPrefixMock.mockReturnValue('some/next/asset/prefix');
      expect(getStaticSrc('/foo')).toEqual(
        '/some/next/asset/prefix/static/foo',
      );
    });
  });

  describe(getDocPageHref, () => {
    it('builds the proper href with query', () => {
      expect(getDocPageHref('/foo/README.md')).toEqual(
        '/?docFilePath=%2Ffoo%2FREADME.md',
      );
    });
  });

  describe(getDocPathRootHref, () => {
    it('builds the proper href with query', () => {
      expect(getDocPathRootHref()).toEqual('/?docFilePath=%2FREADME.md');
    });
  });
});
