/**
 * Manages building and importing document markdown files using dynamic imports.
 * This exists as a separate file from index.ts namely so we can mock this out
 * to making testing possible.
 */

/**
 * We create a require context in order to allow us to efficiently use expressions within dynamic imports.
 * The efficiency part comes from filtering out node_modules through the Regex test when searching for README.md
 * files recursively across the Tachyon project.
 *
 * Regex details: https://regex101.com/r/Y6AJXZ/3
 */

// __WebpackModuleApi requires @types/webpack-env
let documentRequireContext: __WebpackModuleApi.RequireContext | undefined;

if (process.env.NODE_ENV !== 'test') {
  documentRequireContext = require.context(
    'raw-loader!./../../../../../',
    true,
    /^\.\/(?:(?!node_modules)[\w-]+\/)*[\w-]*(?:\.md)$/,
  );
}

/**
 * Call this function to import a Markdown file in the pre-loaded context using raw-loader.
 */
export function requireDocument(docPath: string): string {
  if (!documentRequireContext) {
    return '';
  }

  // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
  return documentRequireContext(`.${docPath}`).default as string;
}

export function getContextDocPaths(): string[] {
  if (!documentRequireContext) {
    return [];
  }

  // strip leading `./` from paths for downstream consumers
  return documentRequireContext
    .keys()
    .map((docPath) => docPath.replace(/^\.\//, ''));
}
