import { memoizeDeepAll, titleCase } from 'tachyon-utils';
import { getContextDocPaths } from './document-context';

export { requireDocument } from './document-context';

export type TopicSet = Record<string, Topic>;

export interface Topic {
  document?: Doc;
  subTopics: TopicSet;
  title: string;
}

export interface Doc {
  filePath: string;
  title: string;
  urlPath: string;
}

export interface DocumentationTopicData {
  documents: Doc[];
  rootTopic: Topic;
}

/**
 * A list of path prefixes that should be used to filter out unwanted documents
 * from the documentation hierarchy.
 */
const IGNORED_DOC_PATH_PREFIXES: string[] = ['pull_request_template'];

export function formatTopicTitle(title: string): string {
  return titleCase(title.replace(/-+/g, ' ').toLowerCase());
}

/**
 * Recursive function for walking a documentation file path a generating tree of
 * the directory structure (accumulated inside/underneath the root DocTopic
 * object passed to the root invocation) and all assocations between doc paths
 * and app paths (accumulated in the passed-in docAppPaths array). All
 * accumulation is done via mutations of the passed-in objects.
 *
 * docPath = the location of a README in the filesystem, starts with `/` and
 *   ends with `.md`
 * urlPath = the corresponding URL path (relative to any prefixing) for a
 *   README, starts and ends with `/`
 */
function buildDocumentationTree(
  [topic, ...subTopics]: string[],
  parentTopic: Topic,
  parentPath: string,
  docAppPaths: Doc[],
): void {
  // strip .md to create topic-slug
  const topicSlug = topic.replace(/\.md$/, '');
  // create path to doc file
  const filePath = `${parentPath}${topic}`;
  // create path to topic in app, adding trailing slash
  const urlPath = `${parentPath}${topicSlug}/`;

  if (topicSlug === 'README') {
    // treat README as the root doc for its enclosing topic
    parentTopic.document = {
      filePath,
      title: parentTopic.title,
      urlPath: parentPath,
    };
    docAppPaths.push(parentTopic.document);
    // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  } else if (!parentTopic.subTopics[topicSlug]) {
    // create a new topic if necessary
    const newTopic: Topic = {
      subTopics: {},
      title: formatTopicTitle(topicSlug),
    };

    if (topic.endsWith('.md')) {
      newTopic.document = { filePath, title: newTopic.title, urlPath };
      docAppPaths.push(newTopic.document);
    }

    parentTopic.subTopics[topicSlug] = newTopic;
  }

  if (subTopics.length) {
    // if subtopics, recurse
    buildDocumentationTree(
      subTopics,
      parentTopic.subTopics[topicSlug],
      urlPath,
      docAppPaths,
    );
  }
}

/**
 * Takes a list of document paths and parses them into a tree of results by
 * grouping common paths together represented as "topics". When a README.md
 * is encountered for a directory it is treated as the "root" document for it.
 * When any other .md extension file is encounted, a new "topic" is generated
 * for it using the name of the file. Also correlates documentation paths with
 * application URL paths.
 *
 * Example:
 * Paths: [README.md, tests/README.md, tests/ADVANCED.md, app/INTERNATIONALIZATION.md]
 *
 * Tree:
 * README.md
 *   --> Tests (README.md)
 *     --> Advanced (ADVANCED.md)
 *   --> App
 *     --> Internationalization (INTERNATIONALIZATION.md)
 */
export function _parseDocumentationTopics(
  documentFilePaths: string[],
  ignoredPrefixes: string[],
): DocumentationTopicData {
  const rootTopic: Topic = {
    subTopics: {},
    title: 'Root',
  };

  const documents: Doc[] = [];

  documentFilePaths
    .filter(
      // Filter out things that would be too painful to deal with via regex (as
      // defined by having a path that starts with an entry in the ignoredPrefixes)
      (filePath) =>
        !ignoredPrefixes.find((ignoredPrefix) =>
          filePath.startsWith(ignoredPrefix),
        ),
    )
    .forEach((path) => {
      buildDocumentationTree(path.split('/'), rootTopic, '/', documents);
    });

  return { documents, rootTopic };
}

/**
 * We memoize this function since:
 * 1) The keys used to compute this should never change.
 * 2) It is expensive to execute.
 * 3) It enables React to be more efficient with shallow pointer object comparisons.
 */
export const parseDocumentationTopics = memoizeDeepAll(
  (): DocumentationTopicData =>
    _parseDocumentationTopics(getContextDocPaths(), IGNORED_DOC_PATH_PREFIXES),
);

export function getDocuments(): Doc[] {
  return parseDocumentationTopics().documents;
}

export function getTopicTree(): Topic {
  return parseDocumentationTopics().rootTopic;
}
